/* eslint-disable no-dupe-class-members */
//
// This class only relays all of the packets from YGOPro server to client (of this server) or vice versa.
//
import { Socket } from "net";

import Packet from "@network/Packet";
import Room from "@game/Room";
import { PacketType } from "@network/message";
import { ServerToClientMessageType } from "@network/message/stoc";

import logger from "@utils/logger";

type PacketListener = (packet: Packet<PacketType.ServerToClient>) => void;

//
// This class is just for `Tunneling` between ygopro server instance and client.
//
export default class Bridge {
    private buffer: Buffer[] = [];
    private socket: Socket | null = null;
    private packetListener: PacketListener | null = null;
    private pipelines: PacketListener[] = [];
    private _hadNewConnection: boolean = false;
    private _closed: boolean = false;

    public get opened() {
        return Boolean(this.socket);
    }
    public get hadNewConnection() {
        return this._hadNewConnection;
    }
    public get closed(): boolean {
        return this._closed;
    }

    /**
     * Connect to a ygopro server instance.
     *
     * @param room
     */
    public open = (room: Room) => {
        return new Promise(resolve => {
            this.socket = new Socket();
            this.socket.on("error", console.error);
            this.socket.on("data", this.onData);
            this.socket.on("close", this.onClose);
            this.socket.connect(room.port, room.host, () => {
                this.flushBuffer();
                resolve();
            });
        });
    };

    public close = (force: boolean = false, newConnection: boolean = false) => {
        if (!this.opened || !this.socket) return;

        if (!force) {
            this.socket.end();
        } else {
            this.socket.destroy();
        }

        this._hadNewConnection = newConnection;
    };

    /**
     * Send series of data from client to ygopro server instance.
     *
     * @param data
     */
    public write = (data: Buffer) => {
        if (!this.opened) {
            // if there's no connection with ygopro server instance, store on the buffer first.
            // the buffer will be flushed when it connected with server.
            this.buffer.push(data);
            return;
        }

        if (this.socket) {
            // refuse if socket is already closed.
            if (!this.socket.writable) {
                return;
            }

            // otherwise just send it to the server directly.
            this.socket.write(data);
        }
    };

    public send(buffer: Buffer): void;
    public send(packet: Packet<PacketType.ClientToServer>): void;
    public send(data: Buffer | Packet<PacketType.ClientToServer>) {
        if (data instanceof Packet) {
            this.write(data.buffer);
            return;
        }

        this.write(data);
    }

    /**
     * Register event listener only for server to client packets.
     *
     * @param packetListener
     */
    public receive = (packetListener: PacketListener) => {
        this.packetListener = packetListener;
    };

    public addPipeline = (pipeline: PacketListener) => {
        this.pipelines.push(pipeline);
    };
    public removePipeline = (pipeline: PacketListener) => {
        this.pipelines = this.pipelines.filter(p => p !== pipeline);
    };
    public removeAllPipelines = () => {
        this.pipelines = [];
    };

    private flushBuffer = () => {
        this.buffer = this.buffer.filter(data => {
            this.write(data);
            return false;
        });
    };
    private onData = (data: Buffer) => {
        if (!this.packetListener) {
            return;
        }

        const packets = Packet.parse(data, PacketType.ServerToClient);
        if (__DEV__) {
            const debugInfo = packets
                .map(packet => {
                    return `[type: ${packet.type} (${ServerToClientMessageType[packet.type]})]`;
                })
                .join(", ");

            logger.debug(`Received ${packets.length} server to client packets: ${debugInfo}`);
        }

        packets.forEach(this.packetListener);

        this.pipelines.forEach(pipeline => {
            packets.forEach(packet => {
                pipeline(packet);
            });
        });
    };
    private onClose = () => {
        this._closed = true;
    };
}
