#!/bin/bash

echo "Cleaning up workspace and dist directories"

rm -rf workspace dist || true
mkdir -p workspace dist/redtexts dist/pics dist/script dist/hashes
echo '# Auto generated by ygopro-custom-free' > dist/strings.conf

# YGOPRO_SERVER_PATH="/home/nanahira/ygo/srvpro/ygopro"
# SIZE_LIMIT_MB=100
# KEEP_PERIOD_DAYS=30

CURRENT_TIME=$(date +%s)

declare -A used_codes
declare -A ocg_codes
declare -A blacklist_files

blacklist_files["script/constant.lua"]=1
blacklist_files["script/utility.lua"]=1
blacklist_files["script/init.lua"]=1
blacklist_files["script/special.lua"]=1
blacklist_files["script/procedure.lua"]=1

loaded_count=0
while read -r code; do
  ocg_codes[$code]=1
  blacklist_files["script/c$code.lua"]=1
  blacklist_files["pics/$code.jpg"]=1
  blacklist_files["pics/$code.png"]=1
  blacklist_files["pics/field/$code.jpg"]=1
  blacklist_files["pics/field/$code.png"]=1
  loaded_count=$((loaded_count + 1))
done < <(sqlite3 cards.cdb 'select id from datas;')
echo "Loaded $loaded_count OCG codes"

something_changed=0
process_repo() {
  repo_url="$1"

  # check if it's empty, or begins with a #, which is a comment
  if [ -z "$repo_url" ] || [[ "$repo_url" == \#* ]]; then
    return
  fi

  repo_name="$(basename $repo_url | sed 's/.git$//')"
  user_name="$(basename "$(dirname "$repo_url")")"
  repo_identifer="$user_name-$repo_name"

  echo "Cloning $repo_url to workspace/$repo_identifer"

  git clone --depth=1 "$repo_url" "workspace/$repo_identifer"

  if [ ! -d "workspace/$repo_identifer" ]; then
    echo "Failed to clone $repo_url"
    echo "[ERROR] $repo_url Failed to clone" | tee dist/warnings.txt
    return
  fi

  rm -rf "workspace/$repo_identifer/sound" || true

  if [ -d "workspace/$repo_identifer/expansions" ]; then
    echo "Moving expansions out in $repo_identifer"
    mv workspace/$repo_identifer/expansions/* workspace/$repo_identifer/
    rm -rf "workspace/$repo_identifer/expansions" || true
  fi

  current_path="$(pwd)"

  # check last commit date
  cd "workspace/$repo_identifer"

    last_commit_timestamp="$(git log -1 --format="%cd" --date=format:'%s')"
    last_commit_hash="$(git log -1 --format="%H")"

    echo "Checking last commit date of $repo_identifer: $(git log -1 --format="%cd")"

    if [ $((CURRENT_TIME - last_commit_timestamp)) -gt $((KEEP_PERIOD_DAYS * 24 * 60 * 60)) ]; then
      echo "Skipping $repo_identifer due to outdated: $((CURRENT_TIME - last_commit_timestamp)) > $((KEEP_PERIOD_DAYS * 24 * 60 * 60))"
      cd "$current_path"
      echo "[ERROR] $repo_url Outdated: $((CURRENT_TIME - last_commit_timestamp)) > $((KEEP_PERIOD_DAYS * 24 * 60 * 60))" | tee dist/warnings.txt
      return
    fi

    rm -rf .git || true

    echo "Checking for blacklist files in $repo_identifer"

    while read -r file; do
      if [ -z "$file" ]; then
        continue
      fi
      if [ -n "${blacklist_files[$file]}" ]; then
        echo "Skipping $repo_identifer due to blacklist file $file"
        cd "$current_path"
        echo "[ERROR] $repo_url Blacklist file: $file" | tee dist/warnings.txt
        return
      fi
    done < <(find .)

  cd "$current_path"

  # check oversize
  size=$(du -sm "workspace/$repo_identifer" | cut -f1)
  echo "Checking size of $repo_identifer: $size MB"
  if [[ $size -gt $SIZE_LIMIT_MB ]]; then
    echo "Skipping $repo_identifer due to oversize: $size MB > $SIZE_LIMIT_MB MB"
    echo "[ERROR] $repo_url Oversize: $size MB > $SIZE_LIMIT_MB MB" | tee dist/warnings.txt
    return
  fi

  echo "Checking for OCG codes in $repo_identifer"

  while read -r code; do
    if [ -z "$code" ]; then
      continue
    fi
    if [ -n "${ocg_codes[$code]}" ]; then
      echo "Skipping $repo_identifer due to using OCG code $code"
      echo "[ERROR] $repo_url Using OCG code: $code" | tee dist/warnings.txt
      return
    fi
  done < <(ls -1 workspace/$repo_identifer/*.cdb | xargs -I '{}' sqlite3 '{}' 'select id from datas;')

  # fetch previous hash
  previous_hash=$(curl -sL "https://cdn02.moecube.com:444/ygopro-custom-free/content/hashes/$repo_identifer.txt" || true)
  if [[ $previous_hash != $last_commit_hash ]]; then
    current_something_changed=1
    echo "Detected changes in $repo_identifer: $previous_hash -> $last_commit_hash"
  else
    current_something_changed=0
    echo "No changes detected in $repo_identifer: $last_commit_hash"
  fi

  echo "Checking for redtexts in $repo_identifer"

  cd "$YGOPRO_SERVER_PATH"

    rm expansions || rmdir expansions || true
    ln -sf "$current_path/workspace/$repo_identifer" expansions

    ls -1 $current_path/workspace/$repo_identifer/*.cdb | xargs -I '{}' -d '\n' sqlite3 '{}' 'select id from datas where type != 0x11 and (type & 0x4000) = 0;' | xargs -I '{}' -d '\n' ./ygopro '{}' 2>&1 | tee "$current_path/dist/redtexts/$repo_identifer.txt"

  cd "$current_path"

  redtext_line_count=$(wc -l dist/redtexts/$repo_identifer.txt | cut -d' ' -f1)

  echo "Finished checking for redtexts in $repo_identifer, $redtext_line_count redtexts found"

  if [ $redtext_line_count -gt 0 ]; then
    echo "Skipping $repo_identifer due to redtexts"
    echo "[ERROR] $repo_url $redtext_line_count Redtexts found" | tee dist/warnings.txt
    return
  fi
  rm dist/redtexts/$repo_identifer.txt

  echo "Checking for used codes in $repo_identifer"

  # add the codes to the used_codes array
  while read -r code; do
    if [ -z "$code" ]; then
      continue
    fi
    if [ -n "${used_codes[$code]}" ]; then
      echo "Code $code in $repo_url is already used in ${used_codes[$code]}"
      echo "[WARNING] CODE CONFLICT: [$code] in ${used_codes[$code]} vs $repo_url" | tee dist/warnings.txt
    else
      used_codes[$code]="$repo_url"
    fi
  done < <(ls -1 workspace/$repo_identifer/*.cdb | xargs -I '{}' -d '\n' sqlite3 '{}' 'select id from datas;')

  echo "Archiving $repo_identifer"

  ls -1 workspace/$repo_identifer/*.cdb | xargs -I '{}' -d '\n' sqlite3 '{}' .dump | sqlite3 dist/custom-free.cdb || true
  # copy strings.conf if necessary
  if [ -f workspace/$repo_identifer/strings.conf ]; then
    echo "" >> dist/strings.conf
    echo "## $repo_identifer" >> dist/strings.conf
    cat workspace/$repo_identifer/strings.conf >> dist/strings.conf
  fi

  cp -rf workspace/$repo_identifer/pics dist/
  cp -rf workspace/$repo_identifer/script dist/
  echo "$last_commit_hash" > dist/hashes/$repo_identifer.txt

  if [[ $current_something_changed == 1 ]]; then
    something_changed=1
  fi

  echo "Finished processing $repo_identifer"
}

# run for each line in each text in repos/*.txt
for file in repos/*.txt; do
  [ -e "$file" ] || continue  # 避免 repos 目录为空时报错
  while read -r repo_url; do
    process_repo "$repo_url"
  done < "$file"
done

if [[ $something_changed == 0 ]] && [[ -n $SCHEDULED ]]; then
  echo "No changes detected, skipping update"
  exit 1
fi

echo "Finished processing all repos"
echo "YGOPro Custom Free Server - Updated at $(date)" > dist/version.txt
