﻿using System.Collections.Generic;
using WindBot;
using WindBot.Game;
using WindBot.Game.AI;
using YGOSharp.OCGWrapper.Enums;

namespace WindBot.Game.AI.Decks
{
    [Deck("Rank V", "AI_Rank5")]
    public class Rank5Executor : DefaultExecutor
    {
        public class CardId
        {
            public static int MistArchfiend = 28601770;
            public static int CyberDragon = 70095154;
            public static int ZWEagleClaw = 29353756;
            public static int SolarWindJammer = 33911264;
            public static int QuickdrawSynchron = 20932152;
            public static int WindUpSoldier = 12299841;
            public static int StarDrawing = 24610207;
            public static int ChronomalyGoldenJet = 88552992;

            public static int InstantFusion = 1845204;
            public static int DoubleSummon = 43422537;
            public static int MysticalSpaceTyphoon = 5318639;
            public static int BookOfMoon = 14087893;
            public static int XyzUnit = 13032689;
            public static int XyzReborn = 26708437;
            public static int MirrorForce = 44095762;
            public static int TorrentialTribute = 53582587;
            public static int XyzVeil = 96457619;

            public static int PanzerDragon = 72959823;
            public static int GaiaDragonTheThunderCharger = 91949988;
            public static int CyberDragonInfinity = 10443957;
            public static int TirasKeeperOfGenesis = 31386180;
            public static int Number61Volcasaurus = 29669359;
            public static int SharkFortress = 50449881;
            public static int CyberDragonNova = 58069384;
        }

        private bool NormalSummoned = false;
        private bool InstantFusionUsed = false;
        private bool DoubleSummonUsed = false;
        private bool CyberDragonInfinitySummoned = false;
        private bool Number61VolcasaurusUsed = false;

        public Rank5Executor(GameAI ai, Duel duel)
            : base(ai, duel)
        {
            // Quick spells
            AddExecutor(ExecutorType.Activate, CardId.BookOfMoon, DefaultBookOfMoon);
            AddExecutor(ExecutorType.Activate, CardId.MysticalSpaceTyphoon, DefaultMysticalSpaceTyphoon);

            // Cyber Dragon Infinity first
            AddExecutor(ExecutorType.SpSummon, CardId.CyberDragonNova, CyberDragonNovaSummon);
            AddExecutor(ExecutorType.Activate, CardId.CyberDragonNova, CyberDragonNovaEffect);
            AddExecutor(ExecutorType.SpSummon, CardId.CyberDragonInfinity, CyberDragonInfinitySummon);
            AddExecutor(ExecutorType.Activate, CardId.CyberDragonInfinity, CyberDragonInfinityEffect);

            // Level 5 monsters without side effects
            AddExecutor(ExecutorType.SpSummon, CardId.CyberDragon);
            AddExecutor(ExecutorType.SpSummon, CardId.SolarWindJammer, SolarWindJammerSummon);
            AddExecutor(ExecutorType.SpSummon, CardId.ZWEagleClaw);
            AddExecutor(ExecutorType.Summon, CardId.ChronomalyGoldenJet, NormalSummon);
            AddExecutor(ExecutorType.Activate, CardId.ChronomalyGoldenJet, ChronomalyGoldenJetEffect);
            AddExecutor(ExecutorType.Summon, CardId.StarDrawing, NormalSummon);
            AddExecutor(ExecutorType.Summon, CardId.WindUpSoldier, NormalSummon);
            AddExecutor(ExecutorType.Activate, CardId.WindUpSoldier, WindUpSoldierEffect);

            // XYZ Monsters: Summon
            AddExecutor(ExecutorType.SpSummon, CardId.Number61Volcasaurus, Number61VolcasaurusSummon);
            AddExecutor(ExecutorType.Activate, CardId.Number61Volcasaurus, Number61VolcasaurusEffect);
            AddExecutor(ExecutorType.SpSummon, CardId.TirasKeeperOfGenesis);
            AddExecutor(ExecutorType.Activate, CardId.TirasKeeperOfGenesis, TirasKeeperOfGenesisEffect);
            AddExecutor(ExecutorType.SpSummon, CardId.SharkFortress);
            AddExecutor(ExecutorType.Activate, CardId.SharkFortress);

            AddExecutor(ExecutorType.SpSummon, CardId.GaiaDragonTheThunderCharger, GaiaDragonTheThunderChargerSummon);


            // Level 5 monsters with side effects
            AddExecutor(ExecutorType.SpSummon, CardId.QuickdrawSynchron, QuickdrawSynchronSummon);
            AddExecutor(ExecutorType.Summon, CardId.MistArchfiend, MistArchfiendSummon);
            AddExecutor(ExecutorType.Activate, CardId.InstantFusion, InstantFusionEffect);

            // Useful spells
            AddExecutor(ExecutorType.Activate, CardId.DoubleSummon, DoubleSummonEffect);
            AddExecutor(ExecutorType.Activate, CardId.XyzUnit, XyzUnitEffect);


            AddExecutor(ExecutorType.Activate, CardId.XyzReborn, XyzRebornEffect);

            AddExecutor(ExecutorType.Activate, CardId.PanzerDragon, PanzerDragonEffect);

            // Reposition
            AddExecutor(ExecutorType.Repos, DefaultMonsterRepos);

            // Set and activate traps
            AddExecutor(ExecutorType.SpellSet, DefaultSpellSet);

            AddExecutor(ExecutorType.Activate, CardId.XyzVeil, XyzVeilEffect);
            AddExecutor(ExecutorType.Activate, CardId.TorrentialTribute, DefaultTorrentialTribute);
            AddExecutor(ExecutorType.Activate, CardId.MirrorForce, DefaultTrap);
        }

        public override bool OnSelectHand()
        {
            return false;
        }

        public override void OnNewTurn()
        {
            NormalSummoned = false;
            InstantFusionUsed = false;
            DoubleSummonUsed = false;
            CyberDragonInfinitySummoned = false;
            Number61VolcasaurusUsed = false;
        }

        private bool NormalSummon()
        {
            NormalSummoned = true;
            return true;
        }

        private bool SolarWindJammerSummon()
        {
            AI.SelectPosition(CardPosition.FaceUpDefence);
            return true;
        }

        private bool QuickdrawSynchronSummon()
        {
            if (!needLV5())
                return false;
            AI.SelectCard(new[]
                {
                    CardId.QuickdrawSynchron,
                    CardId.ZWEagleClaw,
                    CardId.SolarWindJammer,
                    CardId.CyberDragon,
                    CardId.MistArchfiend,
                    CardId.WindUpSoldier,
                    CardId.StarDrawing,
                    CardId.ChronomalyGoldenJet
                });
            return true;
        }

        private bool MistArchfiendSummon()
        {
            if (!needLV5())
                return false;
            AI.SelectOption(1);
            NormalSummoned = true;
            return true;
        }

        private bool InstantFusionEffect()
        {
            if (!needLV5())
                return false;
            InstantFusionUsed = true;
            return true;
        }

        private bool needLV5()
        {
            if (HaveLV5OnField())
                return true;
            int lv5Count = 0;
            IList<ClientCard> hand = Bot.Hand;
            foreach (ClientCard card in hand)
            {
                if (card.Id == CardId.InstantFusion && !InstantFusionUsed)
                    ++lv5Count;
                if (card.Id == CardId.QuickdrawSynchron && Bot.Hand.ContainsMonsterWithLevel(4))
                    ++lv5Count;
                if (card.Id == CardId.MistArchfiend && !NormalSummoned)
                    ++lv5Count;
                if (card.Id == CardId.DoubleSummon && DoubleSummonEffect())
                    ++lv5Count;
            }
            if (lv5Count >= 2)
                return true;
            return false;
        }

        private bool WindUpSoldierEffect()
        {
            return HaveLV5OnField();
        }

        private bool ChronomalyGoldenJetEffect()
        {
            return Card.Level == 4;
        }

        private bool DoubleSummonEffect()
        {
            if (!NormalSummoned || DoubleSummonUsed)
                return false;
            IList<ClientCard> hand = Bot.Hand;
            foreach (ClientCard card in hand)
            {
                if (card.Id == CardId.MistArchfiend ||
                    card.Id == CardId.WindUpSoldier ||
                    card.Id == CardId.StarDrawing ||
                    card.Id == CardId.ChronomalyGoldenJet)
                {
                    NormalSummoned = false;
                    DoubleSummonUsed = true;
                    return true;
                }
            }
            return false;
        }

        private bool CyberDragonNovaSummon()
        {
            return !CyberDragonInfinitySummoned;
        }

        private bool CyberDragonNovaEffect()
        {
            if (ActivateDescription == AI.Utils.GetStringId(CardId.CyberDragonNova, 0))
            {
                return true;
            }
            else if (Card.Location == CardLocation.Grave)
            {
                AI.SelectPosition(CardPosition.FaceUpDefence);
                return true;
            }
            else
            {
                return false;
            }
        }

        private bool CyberDragonInfinitySummon()
        {
            CyberDragonInfinitySummoned = true;
            return true;
        }

        private bool CyberDragonInfinityEffect()
        {
            if (CurrentChain.Count > 0)
            {
                return LastChainPlayer == 1;
            }
            else
            {
                List<ClientCard> monsters = Enemy.GetMonsters();
                ClientCard bestmonster = null;
                foreach (ClientCard monster in monsters)
                {
                    if (monster.IsAttack() && (bestmonster == null || monster.Attack >= bestmonster.Attack))
                        bestmonster = monster;
                }
                if (bestmonster != null)
                {
                    AI.SelectCard(bestmonster);
                    return true;
                }
            }
            return false;
        }

        private bool Number61VolcasaurusSummon()
        {
            return AI.Utils.IsOneEnemyBetterThanValue(2000, false);
        }

        private bool Number61VolcasaurusEffect()
        {
            ClientCard target = AI.Utils.GetProblematicEnemyMonster(2000);
            if (target != null)
            {
                AI.SelectCard(CardId.CyberDragon);
                AI.SelectNextCard(target);
                Number61VolcasaurusUsed = true;
                return true;
            }
            return false;
        }

        private bool TirasKeeperOfGenesisEffect()
        {
            ClientCard target = AI.Utils.GetProblematicEnemyCard();
            if (target != null)
            {
                AI.SelectCard(target);
            }
            return true;
        }

        private bool GaiaDragonTheThunderChargerSummon()
        {
            if (Number61VolcasaurusUsed && Bot.HasInMonstersZone(CardId.Number61Volcasaurus))
            {
                AI.SelectCard(CardId.Number61Volcasaurus);
                return true;
            }
            List<ClientCard> monsters = Bot.GetMonsters();
            foreach (ClientCard monster in monsters)
            {
                if (monster.HasType(CardType.Xyz) && !monster.HasXyzMaterial())
                {
                    AI.SelectCard(monster);
                    return true;
                }
            }
            return false;
        }

        private bool XyzRebornEffect()
        {
            foreach (ClientCard card in Bot.SpellZone)
            {
                if (card != null &&
                    card.Id == Card.Id &&
                    card.HasPosition(CardPosition.FaceUp))
                    return false;
            }
            AI.SelectCard(new[]
                {
                    CardId.CyberDragonInfinity,
                    CardId.CyberDragonNova,
                    CardId.TirasKeeperOfGenesis,
                    CardId.SharkFortress,
                    CardId.Number61Volcasaurus
                });
            return true;
        }

        private bool XyzUnitEffect()
        {
            List<ClientCard> monsters = Bot.GetMonsters();
            return monsters.Exists(p => p.HasType(CardType.Xyz));
        }

        private bool PanzerDragonEffect()
        {
            ClientCard target = AI.Utils.GetBestEnemyCard();
            if (target != null)
            {
                AI.SelectCard(target);
                return true;
            }
            return false;
        }

        private bool XyzVeilEffect()
        {
            List<ClientCard> spells = Bot.GetSpells();
            foreach (ClientCard spell in spells)
            {
                if (spell.Id == CardId.XyzVeil && !spell.IsFacedown())
                    return false;
            }
            List<ClientCard> monsters = Bot.GetMonsters();
            foreach (ClientCard monster in monsters)
            {
                if (monster.HasType(CardType.Xyz))
                    return true;
            }
            return false;
        }

        private bool HaveLV5OnField()
        {
            List<ClientCard> monsters = Bot.GetMonsters();
            foreach (ClientCard monster in monsters)
            {
                if (monster.HasType(CardType.Monster) &&
                    !monster.HasType(CardType.Xyz) &&
                    (monster.Level == 5
                    || monster.Id == CardId.StarDrawing
                    || (monster.Id == CardId.WindUpSoldier) && !monster.Equals(Card)))
                    return true;
            }
            return false;
        }
    }
}
