#!/bin/bash

echo "Cleaning up workspace and dist directories"

rm -rf workspace dist || true
mkdir -p workspace dist/redtexts dist/pics dist/script
echo '# Auto generated by ygopro-custom-free' > dist/strings.conf

# YGOPRO_SERVER_PATH="/home/nanahira/ygo/srvpro/ygopro"
# SIZE_LIMIT_MB=100
# KEEP_PERIOD_DAYS=30

CURRENT_TIME=$(date +%s)

process_repo() {
  repo_url="$1"

  # check if it's empty, or begins with a #, which is a comment
  if [ -z "$repo_url" ] || [[ "$repo_url" == \#* ]]; then
    return
  fi

  repo_name="$(basename $repo_url | sed 's/.git$//')"
  user_name="$(basename "$(dirname "$repo_url")")"
  repo_identifer="$user_name-$repo_name"

  echo "Cloning $repo_url to workspace/$repo_identifer"

  git clone --depth=1 "$repo_url" "workspace/$repo_identifer"

  if [ ! -d "workspace/$repo_identifer" ]; then
    echo "Failed to clone $repo_url"
    return
  fi

  # check oversize
  size=$(du -sm "workspace/$repo_identifer" | cut -f1)
  echo "Checking size of $repo_identifer: $size MB"
  if [[ $size > $SIZE_LIMIT_MB ]]; then
    echo "Skipping $repo_identifer due to oversize: $size MB > $SIZE_LIMIT_MB MB"
    return
  fi

  current_path="$(pwd)"

  # check last commit date
  cd "workspace/$repo_identifer"
  last_commit_timestamp="$(git log -1 --format="%cd" --date=format:'%s')"
  echo "Checking last commit date of $repo_identifer: $(git log -1 --format="%cd")"
  cd "$current_path"

  if [ $((CURRENT_TIME - last_commit_timestamp)) -gt $((KEEP_PERIOD_DAYS * 24 * 60 * 60)) ]; then
    echo "Skipping $repo_identifer due to outdated: $((CURRENT_TIME - last_commit_timestamp)) > $((KEEP_PERIOD_DAYS * 24 * 60 * 60))"
    return
  fi

  echo "Checking for redtexts in $repo_identifer"


  cd "$YGOPRO_SERVER_PATH"
  rm expansions || rmdir expansions || true
  ln -sf "$current_path/workspace/$repo_identifer" expansions

  ls -1 $current_path/workspace/$repo_identifer/*.cdb | xargs -I '{}' sqlite3 '{}' 'select id from datas where type != 0x11 and (type & 0x4000) = 0;' | xargs -I '{}' ./ygopro '{}' 2>&1 | tee "$current_path/dist/redtexts/$repo_identifer.txt"
  cd "$current_path"

  redtext_line_count=$(wc -l dist/redtexts/$repo_identifer.txt | cut -d' ' -f1)

  echo "Finished checking for redtexts in $repo_identifer, $redtext_line_count redtexts found"

  if [ $redtext_line_count -gt 0 ]; then
    echo "Skipping $repo_identifer due to redtexts"
    return
  fi
  rm dist/redtexts/$repo_identifer.txt

  echo "Copying $repo_identifer to dist/$repo_identifer"

  ls -1 workspace/$repo_identifer/*.cdb | xargs -I '{}' sqlite3 '{}' .dump | sqlite3 dist/custom-free.cdb || true
  # copy strings.conf if necessary
  if [ -f workspace/$repo_identifer/strings.conf ]; then
    echo "" >> dist/strings.conf
    echo "## $repo_identifer" >> dist/strings.conf
    cat workspace/$repo_identifer/strings.conf >> dist/strings.conf
  fi
  cp -rf workspace/$repo_identifer/pics dist/
  cp -rf workspace/$repo_identifer/script dist/

  echo "Finished processing $repo_identifer"
}

# run for each line in each text in repos/*.txt
for file in repos/*.txt; do
  [ -e "$file" ] || continue  # 避免 repos 目录为空时报错
  while read -r repo_url; do
    process_repo "$repo_url"
  done < "$file"
done

echo "Finished processing all repos"
echo "Date: $(date)" > dist/version.txt
