﻿using System;
using System.Collections.Generic;
using System.Text;
using DataEditorX.Core;
using DataEditorX.Core.Info;
using System.Drawing;
using System.Drawing.Text;
using DataEditorX.Common;
using System.IO;
using System.Text.RegularExpressions;

namespace DataEditorX.Core.Mse
{
    public class CardMaker : IDisposable
    {
        #region 常量
        public const string TAG = "@";
        public const string VISIBLE_IS_XYZ = "is_xyz";
        public const string VISIBLE_IS_NO_XYZ = "is_no_xyz";
        public const string VISIBLE_IS_MONSTER = "is_monster";
        public const string VISIBLE_IS_NO_MONSTER = "is_no_monster";
        public const string VISIBLE_IS_RACE = "is_race";
        public const string VISIBLE_IS_NO_RACE = "is_no_race";
        public const string VISIBLE_IS_PENDULUM = "is_pendulum";
        public const string VISIBLE_IS_NO_PENDULUM = "is_no_pendulum";
        public const string VISIBLE_IS_WHITE = "is_white";
        public const string VISIBLE_IS_BLACK = "is_black";

        public const string TAG_CARD_TYPE = "@cardtype@";
        public const string TAG_CARD_NAME = "@cardname@";
        public const string TAG_ATTRIBUTE = "@attribute@";
        public const string TAG_LEVEL = "@level@";
        public const string TAG_IMAGE = "@input@";
        //       public const string TAG_NUMBER = "#number";
        public const string TAG_MONSTER_TYPE = "@monstertype@";
        public const string TAG_RULE = "@rule@";
        public const string TAG_ATK = "@atk@";
        public const string TAG_DEF = "@def@";
        public const string TAG_CODE = "@cardcode@";
        public const string TAG_PRULE = "@prule@";
        public const string TAG_PSCALE1 = "@pscale1@";
        public const string TAG_PSCALE2 = "@pscale2@";


        public const string DIR_FONT = "font";
        public const string DIR_STYLE = "style";

        public const int MAX_TYPE = 3;
        #endregion

        private MSEConfig config;
        private string workPath;
        private string stylePath;
        private bool isFirst = true;
        private PrivateFontCollection fonts;//字体缓存
        private Dictionary<string, Bitmap> imageCache;//图片缓存
        private ItemGroup mGroup;//样式

        private Bitmap tmpImage;
        public CardMaker(string path)
        {
            this.workPath = path;
            this.config = new MSEConfig(path);
            stylePath = Path.Combine(workPath, DIR_STYLE);
            fonts = new PrivateFontCollection();
            imageCache = new Dictionary<string, Bitmap>();
            mGroup = new ItemGroup();
        }
        public void Dispose()
        {
            imageCache.Clear();
            fonts.Dispose();
        }

        public void Init(string stylename)
        {
            if(isFirst)
                LoadFont();
            isFirst = false;
            string xmlfile = Path.Combine(stylePath, stylename);
            Read(xmlfile);
        }
        #region 
        public bool setBitmap(Bitmap bmp)
        {
            if (tmpImage == null)
            {
                tmpImage = bmp;
                return true;
            }
            if (tmpImage.GetHashCode() != bmp.GetHashCode())
            {
                tmpImage = bmp;
                return true;
            }
            return false;
        }
        public void setBitmap(string file)
        {
            tmpImage = new Bitmap(file);
        }

        public Bitmap GetBitmap(Card card)
        {
            return GetBitmap(card, tmpImage);
        }
        public Bitmap GetBitmap(Card card, Bitmap input)
        {
            setBitmap(input);
            Decode(card, tmpImage);
            return mGroup.GetBitmap();
        }
        public Bitmap GetBitmap(Card card, string input)
        {
            setBitmap(input);
            return GetBitmap(card, tmpImage);
        }
        #endregion

        #region 加载样式
        //加载样式
        public void Read(string file)
        {
            mGroup = (ItemGroup)XMLHelper
                .DeserializeFromXml<ItemGroup>(file);
            foreach (Item item in mGroup.items)
            {
                //初始化字体
                item.font.font = new Font(item.font.fontName,
                    item.font.fontSize,
                    item.font.fontStyle);
                //加载不变的图片
                string img = item.src;
                if (string.IsNullOrEmpty(img))
                    continue;
                else if (img.Contains(TAG))//是选择型图片
                {
                    foreach (Choice ch in item.srcs)
                    {
                        if (!imageCache.ContainsKey(ch.val))
                        {//加载到内存
                            string imgfile = Path.Combine(stylePath, ch.val);
                            if(File.Exists(imgfile))
                                imageCache.Add(ch.val, new Bitmap(imgfile));
                        }
                    }
                }
                else//固定型图片
                {
                    string imgfile = Path.Combine(stylePath, img);
                    if (File.Exists(imgfile))
                        item.bmp = new Bitmap(imgfile);
                }

            }
        }
        #endregion

        #region 加载字体
        //加载字体
        public void LoadFont()
        {
            string fontpath = Path.Combine(workPath, DIR_FONT);
            if (!Directory.Exists(fontpath))
                return;
            string[] files = Directory.GetFiles(fontpath);
            foreach (string file in files)
            {
                if (file.EndsWith(".ttf", StringComparison.OrdinalIgnoreCase)
                    || file.EndsWith(".ttc", StringComparison.OrdinalIgnoreCase))
                {
                    fonts.AddFontFile(file);
                }
            }
        }
        #endregion

        #region 解析
        /// <summary>
        /// 解析
        /// </summary>
        /// <param name="group">元素集合</param>
        /// <param name="bmp">裁剪的图片</param>
        /// <param name="imageCache">图片缓存</param>
        public void Decode(Card card,Bitmap bmp)
        {
            foreach (Item item in mGroup.items)
            {
                //设置可见度
                item.visible = getVisible(card, item.visibleStr);
                //替换文字
                item.text = getText(card, item.content);
                //裁剪图片
                string src = item.src;
                if (!src.Contains(TAG))//图片是固定的
                    continue;
                if (TAG_IMAGE.Equals(src))
                    item.bmp = bmp;
                else
                {//选择图片
                    src = getImageSrc(card, src, item.srcs);
                    if (imageCache.ContainsKey(src))
                        item.bmp = imageCache[src];
                    else
                        item.bmp = null;
                }
            }
        }
        public static string getImageSrc(Card c, string src, 
            List<Choice> srcs)
        {
            long key = 0;
            if (TAG_CARD_TYPE.Equals(src))
                key = c.type;
            else if(TAG_ATTRIBUTE.Equals(src))
                key = c.attribute;
            else if (TAG_LEVEL.Equals(src))
                key = c.level & 0xff;
            foreach (Choice ch in srcs)
            {
                if (ch.Key == key || (ch.Key & key ) == ch.Key)
                {
                    return ch.val;
                }
            }
            return "";
        }

        #endregion

        #region 转换
        public static string GetDesc(string cdesc, string regx)
        {
            string desc = cdesc;
            desc = desc.Replace("\r\n", "\n");
            desc = desc.Replace("\r", "\n");
            Regex regex = new Regex(regx, RegexOptions.Multiline);
            Match mc = regex.Match(desc);
            if (mc.Success)
                return ((mc.Groups.Count > 1) ?
                    mc.Groups[1].Value : mc.Groups[0].Value)
                    .Trim('\n').Replace("\n", "\n\t\t");
            return "";
        }
       
        public static string GetRace(long race, MSEConfig config)
        {
            if (config.raceDic.ContainsKey(race))
                return config.raceDic[race].Trim();
            return race.ToString("x");
        }
        public static string GetType(CardType ctype, MSEConfig config)
        {
            long type = (long)ctype;
            if (config.typeDic.ContainsKey(type))
                return config.typeDic[type].Trim();
            return type.ToString("x");
        }
        public static string[] GetTypes(Card c, MSEConfig config)
        {
            //卡片类型，效果1，效果2，效果3
            string[] types = new string[] { 
                MseCardType.CARD_NORMAL, "", "", "" };
            if (c.IsType(CardType.TYPE_MONSTER))
            {//卡片类型和第1效果
                if (c.IsType(CardType.TYPE_XYZ))
                {
                    types[0] = MseCardType.CARD_XYZ;
                    types[1] = GetType(CardType.TYPE_XYZ, config);
                }
                else if (c.IsType(CardType.TYPE_TOKEN))
                {
                    types[0] = (c.race == 0) ?
                        MseCardType.CARD_TOKEN2
                        : MseCardType.CARD_TOKEN;
                    types[1] = GetType(CardType.TYPE_TOKEN, config);
                }
                else if (c.IsType(CardType.TYPE_RITUAL))
                {
                    types[0] = MseCardType.CARD_RITUAL;
                    types[1] = GetType(CardType.TYPE_RITUAL, config);
                }
                else if (c.IsType(CardType.TYPE_FUSION))
                {
                    types[0] = MseCardType.CARD_FUSION;
                    types[1] = GetType(CardType.TYPE_FUSION, config);
                }
                else if (c.IsType(CardType.TYPE_SYNCHRO))
                {
                    types[0] = MseCardType.CARD_SYNCHRO;
                    types[1] = GetType(CardType.TYPE_SYNCHRO, config);
                }
                else if (c.IsType(CardType.TYPE_EFFECT))
                {
                    types[0] = MseCardType.CARD_EFFECT;
                }
                else
                    types[0] = MseCardType.CARD_NORMAL;
                //同调
                if (types[0] == MseCardType.CARD_SYNCHRO
                    || types[0] == MseCardType.CARD_TOKEN)
                {
                    if (c.IsType(CardType.TYPE_TUNER)
                       && c.IsType(CardType.TYPE_EFFECT))
                    {//调整效果
                        types[2] = GetType(CardType.TYPE_TUNER, config);
                        types[3] = GetType(CardType.TYPE_EFFECT, config);
                    }
                    else if (c.IsType(CardType.TYPE_TUNER))
                    {
                        types[2] = GetType(CardType.TYPE_TUNER, config);
                    }
                    else if (c.IsType(CardType.TYPE_EFFECT))
                    {
                        types[2] = GetType(CardType.TYPE_EFFECT, config);
                    }
                }
                else if (types[0] == MseCardType.CARD_NORMAL)
                {
                    if (c.IsType(CardType.TYPE_PENDULUM))//灵摆
                        types[1] = GetType(CardType.TYPE_PENDULUM, config);
                    else if (c.IsType(CardType.TYPE_TUNER))//调整
                        types[1] = GetType(CardType.TYPE_TUNER, config);
                }
                else if (types[0] != MseCardType.CARD_EFFECT)
                {//效果
                    if (c.IsType(CardType.TYPE_EFFECT))
                        types[2] = GetType(CardType.TYPE_EFFECT, config);
                }
                else
                {//效果怪兽
                    types[2] = GetType(CardType.TYPE_EFFECT, config);
                    if (c.IsType(CardType.TYPE_PENDULUM))
                        types[1] = GetType(CardType.TYPE_PENDULUM, config);
                    else if (c.IsType(CardType.TYPE_TUNER))
                        types[1] = GetType(CardType.TYPE_TUNER, config);
                    else if (c.IsType(CardType.TYPE_SPIRIT))
                        types[1] = GetType(CardType.TYPE_SPIRIT, config);
                    else if (c.IsType(CardType.TYPE_TOON))
                        types[1] = GetType(CardType.TYPE_TOON, config);
                    else if (c.IsType(CardType.TYPE_UNION))
                        types[1] = GetType(CardType.TYPE_UNION, config);
                    else if (c.IsType(CardType.TYPE_DUAL))
                        types[1] = GetType(CardType.TYPE_DUAL, config);
                    else if (c.IsType(CardType.TYPE_FLIP))
                        types[1] = GetType(CardType.TYPE_FLIP, config);
                    else
                    {
                        types[1] = GetType(CardType.TYPE_EFFECT, config);
                        types[2] = "";
                    }
                }

            }
            if (c.race == 0)//如果没有种族
            {
                types[1] = "";
                types[2] = "";
            }
            return types;
        }
        public static string getMonsterType(Card c, MSEConfig config)
        {
            string[] types = GetTypes(c, config);
            string text = GetRace(c.race, config);
            for (int i = 1; i <= MAX_TYPE; i++)
            {
                string type =types[i];
                if (!string.IsNullOrEmpty(type))
                    text += config.SEP + type;
            }
            return config.bracketLeft + text + config.bracketRight;
        }
        #endregion

        #region 替换文字
        string getText(Card card, string content)
        {
            if (TAG_CARD_NAME.Equals(content))
            {
                return card.name;
            }
            else if (TAG_MONSTER_TYPE.Equals(content))
            {
                return card.IsType(CardType.TYPE_MONSTER) ? 
                    getMonsterType(card, config) : "";
            }
            else if (TAG_RULE.Equals(content))
            {
                return card.IsType(CardType.TYPE_PENDULUM) ?
                    GetDesc(card.desc, config.regx_monster)
                    :card.desc;
            }
            else if (TAG_PRULE.Equals(content))
            {
                return GetDesc(card.desc, config.regx_pendulum);
            }
            else if (TAG_ATK.Equals(content))
            {
                return (card.atk < 0) ? "?" : card.atk.ToString();
            }
            else if (TAG_DEF.Equals(content))
            {
                return (card.def < 0) ? "?" : card.def.ToString();
            }
            else if (TAG_CODE.Equals(content))
            {
                return card.idString;
            }
            else if (TAG_PSCALE1.Equals(content))
            {
                return card.PScaleLeft().ToString();
            }
            else if (TAG_PSCALE2.Equals(content))
            {
                return card.PScaleRight().ToString();
            }
            else
            {
                return "";
            }
        }
        #endregion

        #region 可见度
        static bool getVisible(Card card, string str)
        {
            if (VISIBLE_IS_XYZ.Equals(str))
            {
                return card.IsType(CardType.TYPE_XYZ);
            }
            else if (VISIBLE_IS_NO_XYZ.Equals(str))
            {
                return !card.IsType(CardType.TYPE_XYZ) 
                    && card.IsType(CardType.TYPE_MONSTER);
            }
            else if (VISIBLE_IS_MONSTER.Equals(str))
            {
                return card.IsType(CardType.TYPE_MONSTER);
            }
            else if (VISIBLE_IS_NO_MONSTER.Equals(str))
            {
                return !card.IsType(CardType.TYPE_MONSTER);
            }
            else if (VISIBLE_IS_RACE.Equals(str))
            {
                return (card.race > 0);
            }
            else if (VISIBLE_IS_NO_RACE.Equals(str))
            {
                return (card.race <= 0);
            }
            else if (VISIBLE_IS_PENDULUM.Equals(str))
            {
                return card.IsType(CardType.TYPE_PENDULUM);
            }
            else if (VISIBLE_IS_NO_PENDULUM.Equals(str))
            {
                return !card.IsType(CardType.TYPE_PENDULUM)
                    && card.IsType(CardType.TYPE_MONSTER);
            }
            else if (VISIBLE_IS_WHITE.Equals(str))
            {
                return card.IsType(CardType.TYPE_XYZ)
                    || !card.IsType(CardType.TYPE_MONSTER);
            }
            else if (VISIBLE_IS_BLACK.Equals(str))
            {
                return card.IsType(CardType.TYPE_MONSTER)
                    && !card.IsType(CardType.TYPE_XYZ);
            }
            else
            {
                return true;
            }
        }
        #endregion
    }
}
