import { Button, Input } from "antd";
import { useEffect, useState } from "react";
import { useSnapshot } from "valtio";

import { sendChat } from "@/api";
import { chatStore, roomStore } from "@/stores";
import { IconFont, ScrollableArea } from "@/ui/Shared";

import styles from "./Chat.module.scss";

interface ChatItem {
  name: string;
  time: string;
  content: string;
}

export const Chat: React.FC = () => {
  const [chatlist, setChatlist] = useState<ChatItem[]>([]);
  const [input, setInput] = useState<string | undefined>(undefined);
  const chat = useSnapshot(chatStore);

  useEffect(() => {
    if (chatStore.sender >= 0 && chatStore.message.length != 0) {
      const now = new Date();
      const sender = chatStore.sender;
      const name =
        sender < roomStore.players.length
          ? roomStore.players[sender]?.name ?? "?"
          : (sender > 8 && sender < 11) || sender > 19
          ? "?"
          : "System";
      setChatlist((prev) => [
        ...prev,
        {
          name: name,
          time: `${now.getHours()}:${now.getMinutes()}:${now.getSeconds()}`,
          content: chatStore.message,
        },
      ]);
    }
  }, [chat]);

  return (
    <div className={styles.chat}>
      <ScrollableArea className={styles.dialogs}>
        {chatlist.map((item, idx) => (
          <DialogItem key={idx} {...item} />
        ))}
      </ScrollableArea>
      <div className={styles.input}>
        <Input.TextArea
          bordered={false}
          value={input}
          onChange={(event) => setInput(event.target.value)}
          autoSize
          placeholder="请输入聊天内容"
        />
        <Button
          type="text"
          icon={<IconFont type="icon-send" size={16} />}
          onClick={() => {
            if (input !== undefined) {
              sendChat(input);
              setInput(undefined);
            }
          }}
        />
      </div>
    </div>
  );
};

const DialogItem: React.FC<ChatItem> = ({ name, time, content }) => {
  return (
    <div className={styles.item}>
      <div className={styles.name}>
        {name}
        <span className={styles.time}>{time}</span>
      </div>
      <div className={styles.content}>{content}</div>
    </div>
  );
};
