import {
  CheckOutlined,
  DeleteOutlined,
  EditOutlined,
  FilterOutlined,
  SearchOutlined,
  SortAscendingOutlined,
  UndoOutlined,
} from "@ant-design/icons";
import {
  App,
  Badge,
  Button,
  ConfigProvider,
  Dropdown,
  Input,
  type MenuProps,
  Space,
  type ThemeConfig,
} from "antd";
import classNames from "classnames";
import { memo, useEffect, useRef, useState } from "react";
import { DndProvider, useDrag, useDrop } from "react-dnd";
import { HTML5Backend } from "react-dnd-html5-backend";
import { LoaderFunction } from "react-router-dom";
import { v4 as v4uuid } from "uuid";
import { useSnapshot } from "valtio";
import { subscribeKey } from "valtio/utils";

import { type CardMeta, searchCards } from "@/api";
import { isExtraDeckCard, isToken } from "@/common";
import { FtsConditions } from "@/middleware/sqlite/fts";
import { deckStore, forbiddenStore, type IDeck, initStore } from "@/stores";
import {
  Background,
  IconFont,
  Loading,
  ScrollableArea,
  YgoCard,
} from "@/ui/Shared";

import { CardDetail } from "./CardDetail";
import { DeckSelect } from "./DeckSelect";
import { editDeckStore } from "./editDeckStore";
import { Filter } from "./Filter";
import styles from "./index.module.scss";
import {
  canAdd,
  editingDeckToIDeck,
  iDeckToEditingDeck,
  type Type,
} from "./utils";

const theme: ThemeConfig = {
  components: {
    Layout: {
      colorBgBody: "transparent",
    },
  },
};

export const loader: LoaderFunction = async () => {
  // 必须先加载卡组，不然页面会崩溃
  if (!initStore.decks) {
    await new Promise<void>((rs) => {
      subscribeKey(initStore, "decks", (done) => done && rs());
    });
  }

  // 加载禁限卡表
  await forbiddenStore.init();
  return null;
};

export const Component: React.FC = () => {
  const snapDecks = useSnapshot(deckStore);
  const { sqlite } = useSnapshot(initStore);
  const [selectedDeck, setSelectedDeck] = useState<IDeck>(deckStore.decks[0]);

  const { message } = App.useApp();

  const handleDeckEditorReset = async () => {
    editDeckStore.set(await iDeckToEditingDeck(selectedDeck));
    message.info("重置成功");
  };

  const handleDeckEditorSave = async () => {
    const tmpIDeck = editingDeckToIDeck(editDeckStore);
    const result = await deckStore.update(selectedDeck.deckName, tmpIDeck);
    if (result) {
      setSelectedDeck(tmpIDeck);
      message.info("保存成功");
      editDeckStore.edited = false;
    } else {
      editDeckStore.set(await iDeckToEditingDeck(selectedDeck));
      message.error("保存失败");
      editDeckStore.edited = false;
    }
  };

  return (
    <DndProvider backend={HTML5Backend}>
      <ConfigProvider theme={theme}>
        <Background />
        <div className={styles.layout} style={{ width: "100%" }}>
          <div className={styles.sider}>
            <ScrollableArea className={styles["deck-select-container"]}>
              <DeckSelect
                decks={snapDecks.decks}
                selected={selectedDeck.deckName}
                onSelect={(name) =>
                  setSelectedDeck(deckStore.get(name) ?? deckStore.decks[0])
                }
                onDelete={(id) => console.log(id)}
                onDownload={(id) => console.log(id)}
                onAdd={() => console.log("add")}
              />
            </ScrollableArea>
            <CardDetail code={123} open={false} onClose={() => {}} />
          </div>
          <div className={styles.content}>
            <div className={styles.deck}>
              <DeckEditor
                deck={selectedDeck}
                onClear={editDeckStore.clear}
                onReset={handleDeckEditorReset}
                onSave={handleDeckEditorSave}
              />
            </div>
            <div className={styles.select}>
              {sqlite.progress === 1 ? (
                <Search />
              ) : (
                <div className={styles.container}>
                  <Loading />
                </div>
              )}
            </div>
          </div>
        </div>
      </ConfigProvider>
    </DndProvider>
  );
};
Component.displayName = "Build";

/** 正在编辑的卡组 */
const DeckEditor: React.FC<{
  deck: IDeck;
  onClear: () => void;
  onReset: () => void;
  onSave: () => void;
}> = ({ deck, onClear, onReset, onSave }) => {
  const snapEditDeck = useSnapshot(editDeckStore);
  useEffect(() => {
    iDeckToEditingDeck(deck).then(editDeckStore.set);
  }, [deck]);
  return (
    <div className={styles.container}>
      <Space className={styles.title}>
        <Input
          placeholder="请输入卡组名字"
          bordered={false}
          prefix={<EditOutlined />}
          style={{ width: 400 }}
          onChange={(e) =>
            editDeckStore.set({
              ...editDeckStore,
              deckName: e.target.value,
            })
          }
          value={snapEditDeck.deckName}
        />
        <Space style={{ marginRight: 6 }}>
          <Button
            type="text"
            size="small"
            icon={<DeleteOutlined />}
            onClick={onClear}
          >
            清空
          </Button>
          <Button
            type="text"
            size="small"
            icon={<UndoOutlined />}
            onClick={() => onReset()}
          >
            重置
          </Button>
          <Button
            type={snapEditDeck.edited ? "primary" : "text"}
            size="small"
            icon={<CheckOutlined />}
            onClick={() => onSave()}
          >
            保存
          </Button>
        </Space>
      </Space>
      <ScrollableArea className={styles["deck-zone"]}>
        {(["main", "extra", "side"] as const).map((type) => (
          <DeckZone key={type} type={type} />
        ))}
      </ScrollableArea>
    </div>
  );
};

/** 卡片库，选择卡片加入正在编辑的卡组 */
const Search: React.FC = () => {
  const { modal } = App.useApp();
  const [searchWord, setSearchWord] = useState("");
  const [searchConditions, setSearchConditions] = useState<FtsConditions>({});
  const [searchResult, setSearchResult] = useState<CardMeta[]>([]);

  const sortRef = useRef<(a: CardMeta, b: CardMeta) => number>(
    (a: CardMeta, b: CardMeta) => a.id - b.id
  );

  const setSortRef = (sort: (a: CardMeta, b: CardMeta) => number) => {
    sortRef.current = sort;
    setSearchResult([...searchResult.sort(sortRef.current)]);
  };

  const dropdownOptions: MenuProps["items"] = (
    [
      ["从新到旧", () => setSortRef((a, b) => b.id - a.id)],
      ["从旧到新", () => setSortRef((a, b) => a.id - b.id)],
      [
        "攻击力从高到低",
        () => setSortRef((a, b) => (b.data?.atk ?? 0) - (a.data?.atk ?? 0)),
      ],
      [
        "攻击力从低到高",
        () => setSortRef((a, b) => (a.data?.atk ?? 0) - (b.data?.atk ?? 0)),
      ],
      [
        "守备力从高到低",
        () => setSortRef((a, b) => (b.data?.def ?? 0) - (a.data?.def ?? 0)),
      ],
      [
        "守备力从低到高",
        () => setSortRef((a, b) => (a.data?.def ?? 0) - (b.data?.def ?? 0)),
      ],
      [
        "星/阶/刻/Link从高到低",
        () => setSortRef((a, b) => (a.data?.level ?? 0) - (b.data?.level ?? 0)),
      ],
      [
        "星/阶/刻/Link从低到高",
        () => setSortRef((a, b) => (b.data?.level ?? 0) - (a.data?.level ?? 0)),
      ],
    ] as const
  ).map(([label, onClick], key) => ({ key, label, onClick }));

  const handleSearch = async () => {
    const result = (
      await searchCards({ query: searchWord, conditions: searchConditions })
    )
      .filter((card) => !isToken(card.data.type ?? 0))
      .sort(sortRef.current); // 衍生物不显示
    setSearchResult(result);
  };

  const [_, dropRef] = useDrop({
    accept: ["Card"], // 指明该区域允许接收的拖放物。可以是单个，也可以是数组
    // 里面的值就是useDrag所定义的type
    // 当拖拽物在这个拖放区域放下时触发,这个item就是拖拽物的item（拖拽物携带的数据）
    drop: ({ value, source }: { value: CardMeta; source: Type | "search" }) => {
      if (source !== "search") {
        editDeckStore.remove(source, value);
      }
    },
  });
  return (
    <div className={styles.container} ref={dropRef}>
      <div className={styles.title}>
        <Input
          placeholder="搜索卡片"
          bordered={false}
          suffix={
            <Button
              type="text"
              icon={<SearchOutlined />}
              onClick={handleSearch}
            />
          }
          value={searchWord}
          onChange={(e) => setSearchWord(e.target.value)}
          onKeyUp={(e) => e.key === "Enter" && handleSearch()}
          allowClear
        />
      </div>
      <div className={styles["select-btns"]}>
        <Button
          block
          type="text"
          icon={<FilterOutlined />}
          onClick={() => {
            const { destroy } = modal.info({
              width: 500,
              centered: true,
              title: null,
              icon: null,
              content: (
                <Filter
                  conditions={searchConditions}
                  onConfirm={(newConditions) => {
                    setSearchConditions(newConditions);
                    destroy();
                  }}
                  onCancel={() => destroy()}
                />
              ),
              footer: null,
            });
          }}
        >
          筛选
          {/* TODO: 下面这个Badge应根据有无筛选规则而显示 */}
          {false && <Badge dot offset={[5, -5]} />}
        </Button>
        <Dropdown
          menu={{ items: dropdownOptions }}
          trigger={["click"]}
          placement="bottom"
          arrow
        >
          <Button block type="text" icon={<SortAscendingOutlined />}>
            <span>
              排列
              <span className={styles["search-count"]}>
                ({searchResult.length})
              </span>
            </span>
            {false && <Badge dot offset={[5, -5]} />}
          </Button>
        </Dropdown>
        <Button block type="text" icon={<DeleteOutlined />}>
          重置
        </Button>
      </div>
      <ScrollableArea className={styles["search-cards-container"]}>
        {searchResult.length ? (
          <SearchResults results={searchResult} />
        ) : (
          <div className={styles.empty}>
            <IconFont type="icon-empty" size={40} />
            <div>无搜索结果</div>
          </div>
        )}
      </ScrollableArea>
    </div>
  );
};

/** 正在组卡的zone，包括main/extra/side */
const DeckZone: React.FC<{
  type: Type;
}> = ({ type }) => {
  const { message } = App.useApp();
  const cards = useSnapshot(editDeckStore)[type];
  const [allowToDrop, setAllowToDrop] = useState(false);
  const [{ isOver }, dropRef] = useDrop({
    accept: ["Card"], // 指明该区域允许接收的拖放物。可以是单个，也可以是数组
    // 里面的值就是useDrag所定义的type
    // 当拖拽物在这个拖放区域放下时触发,这个item就是拖拽物的item（拖拽物携带的数据）
    drop: ({ value, source }: { value: CardMeta; source: Type | "search" }) => {
      if (type === source) return;
      const { result, reason } = canAdd(value, type, editDeckStore);
      if (result) {
        editDeckStore.add(type, value);
        if (source !== "search") {
          editDeckStore.remove(source, value);
        }
      } else {
        message.error(reason);
      }
    },
    hover: ({ value, source }) => {
      setAllowToDrop(
        type !== source ? canAdd(value, type, editDeckStore).result : true
      );
    },
    collect: (monitor) => ({
      isOver: monitor.isOver(),
    }),
  });
  return (
    <div
      className={classNames(styles[type], {
        [styles.over]: isOver,
        [styles["not-allow-to-drop"]]: isOver && !allowToDrop,
      })}
      ref={dropRef}
    >
      <div className={styles["card-continer"]}>
        {cards.map((card) => (
          <Card
            value={card}
            key={v4uuid()}
            source={type}
            onRightClick={() => editDeckStore.remove(type, card)}
          />
        ))}
        <div className={styles["editing-zone-name"]}>{type.toUpperCase()}</div>
      </div>
    </div>
  );
};

/** 搜索区的搜索结果，使用memo避免重复渲染 */
const SearchResults: React.FC<{
  results: CardMeta[];
}> = memo(({ results }) => {
  const handleClick = (card: CardMeta) => {
    const type = isExtraDeckCard(card.data.type ?? 0) ? "extra" : "main";
    canAdd(card, type, editDeckStore).result && editDeckStore.add(type, card);
  };
  return (
    <div className={styles["search-cards"]}>
      {results.map((card) => (
        <Card
          value={card}
          key={card.id}
          source="search"
          onClick={() => handleClick(card)}
        />
      ))}
    </div>
  );
});

/** 本组件内使用的单张卡片，增加了文字在图片下方 */
const Card: React.FC<{
  value: CardMeta;
  source: Type | "search";
  onClick?: () => void;
  onRightClick?: () => void;
}> = memo(({ value, source, onClick, onRightClick }) => {
  const ref = useRef<HTMLDivElement>(null);
  const [{ isDragging }, drag] = useDrag({
    type: "Card",
    item: { value, source },
    collect: (monitor: any) => ({
      isDragging: monitor.isDragging(),
    }),
  });
  drag(ref);
  return (
    <div
      className={styles.card}
      ref={ref}
      style={{ opacity: isDragging && source !== "search" ? 0 : 1 }}
      onClick={onClick}
      onContextMenu={(e) => {
        e.preventDefault();
        onRightClick?.();
      }}
    >
      <div className={styles.cardname}>{value.text.name}</div>
      <YgoCard className={styles.cardcover} code={value.id} />
    </div>
  );
});
