import { AfterViewInit, Component } from '@angular/core';
import { ToastService } from './toast.service';
import { lastValueFrom } from 'rxjs';
import * as moment from 'moment';
import {
  Ad,
  BlacklistAccountResultDto,
  WhitelistAccountResultDto,
  WhitelistAccountRefResultDto,
} from './api/models';
import {
  AdService,
  BlacklistAccountService,
  WhitelistAccountService,
} from './api/services';

@Component({
  selector: 'app-root',
  templateUrl: './app.component.html',
  styleUrls: ['./app.component.css'],
})
export class AppComponent implements AfterViewInit {
  search = '';
  displayingAccount = '';
  title = '蔷蔷查询';
  ad: Ad | null = null;
  blacklistResult?: BlacklistAccountResultDto[];
  whitelistResult?: WhitelistAccountResultDto[];
  whitelistDisplayResult?: Array<{
    item: WhitelistAccountResultDto | WhitelistAccountRefResultDto;
    accounts: string[];
    twitters: string[];
  }>;
  constructor(
    public toast: ToastService,
    private adService: AdService,
    private blacklistAccountService: BlacklistAccountService,
    private whitelistAccountService: WhitelistAccountService
  ) {}

  ngAfterViewInit() {
    this.initAd();
  }

  displayDate(date: string) {
    return moment(date).format('YYYY-MM-DD HH:mm:ss');
  }

  async initAd() {
    try {
      const adId = new URLSearchParams(window.location.search).get('ad');
      const { data: ad } = await lastValueFrom(
        this.adService.adControllerGetAd(
          adId ? { id: parseInt(adId) } : undefined
        )
      );
      if (ad) {
        this.ad = ad;
      }
    } catch (e) {
      console.log(`Failed to load ad: ${(e as Error).toString()}`);
    }
  }

  openInNewTab(href: string) {
    Object.assign(document.createElement('a'), {
      target: '_blank',
      rel: 'noopener noreferrer',
      href,
    }).click();
  }

  async onAdClick() {
    const clickResult = await lastValueFrom(
      this.adService.adControllerClickAd({ id: this.ad!.id! })
    );
    if (clickResult?.data) {
      this.openInNewTab(clickResult.data);
    }
  }

  getTwitterLink(twitter: string) {
    const urlObj = new URL('https://x.com');
    urlObj.pathname = `/${twitter}`;
    return urlObj.toString();
  }

  async searchWhitelist(account?: string) {
    this.whitelistResult = undefined;
    this.whitelistDisplayResult = undefined;
    const { data } = await lastValueFrom(
      this.whitelistAccountService.whitelistAccountControllerFindAll({
        recordsPerPage: 5,
        account: account || undefined,
        random: account ? 0 : 1,
      })
    );
    this.whitelistResult = data;
    this.whitelistDisplayResult = this.buildWhitelistDisplayResult(data);
  }

  async checkSearch(allowEmpty = false) {
    const search = this.search.trim();
    if (!search && !allowEmpty) {
      this.toast.warn('不能为空。');
      return false;
    } else if (!search && allowEmpty) {
      return true;
    }
    if (search.length > 10) {
      this.toast.warn('不能超过 10 个字符。');
      return false;
    }
    if (!search.match(/^\d+$/)) {
      this.toast.warn('只能输入数字。');
      return false;
    }
    return true;
  }

  async onBlacklistSearch() {
    this.blacklistResult = undefined;
    this.whitelistResult = undefined;
    this.whitelistDisplayResult = undefined;
    const search = this.search.trim();
    if (!(await this.checkSearch())) {
      return;
    }
    try {
      const { data } = await lastValueFrom(
        this.blacklistAccountService.blacklistAccountControllerFindAll({
          recordsPerPage: 100,
          account: search,
        })
      );
      this.blacklistResult = data;
      if (!data?.length) {
        await this.searchWhitelist(search);
      }
      this.displayingAccount = this.search;
    } catch (e) {
      this.toast.error(`查询失败：${(e as Error).toString()}`);
    }
  }

  async onWhitelistSearch() {
    this.blacklistResult = undefined;
    this.whitelistResult = undefined;
    this.whitelistDisplayResult = undefined;
    const search = this.search.trim();
    if (!(await this.checkSearch())) {
      return;
    }

    try {
      await this.searchWhitelist(search);
      this.displayingAccount = this.search;
    } catch (e) {
      this.toast.error(`查询失败：${(e as Error).toString()}`);
    }
  }
  async onRandomSearch() {
    this.blacklistResult = undefined;
    this.whitelistResult = undefined;
    this.whitelistDisplayResult = undefined;

    try {
      await this.searchWhitelist();
      this.displayingAccount = '';
    } catch (e) {
      this.toast.error(`查询失败：${(e as Error).toString()}`);
    }
  }

  truncateMiddle(text: string, maxLength: number): string {
    if (!text || text.length <= maxLength) {
      return text;
    }
    const keep = Math.floor((maxLength - 3) / 2); // -3 for '...'
    return text.slice(0, keep) + '...' + text.slice(-keep);
  }

  private buildWhitelistDisplayResult(data?: WhitelistAccountResultDto[]):
    | Array<{
        item: WhitelistAccountResultDto | WhitelistAccountRefResultDto;
        accounts: string[];
        twitters: string[];
      }>
    | undefined {
    if (!data) {
      return undefined;
    }
    return data.map((entry) => {
      const item = entry.ref ?? entry;
      const accounts = this.uniqueNonEmptyStrings([
        item.account,
        ...(item.refs?.map((refEntry) => refEntry.account) || []),
      ]);
      const twitters = this.uniqueNonEmptyStrings([
        item.twitter,
        ...(item.refs?.map((refEntry) => refEntry.twitter) || []),
      ]).filter((s) => s && !s.includes('无'));
      return { item, accounts, twitters };
    });
  }

  private uniqueNonEmptyStrings(values: Array<string | undefined>): string[] {
    const seen = new Set<string>();
    for (const value of values) {
      const normalized = value?.trim();
      if (!normalized) {
        continue;
      }
      seen.add(normalized);
    }
    return Array.from(seen);
  }
}
