import 'reflect-metadata';
import { ArrayValueMap, GenericMap, Key, MergeKey, StringDict } from './def';

export type AllDecorators = MethodDecorator &
  ClassDecorator &
  PropertyDecorator;

export class Metadata<M extends StringDict, AM extends StringDict> {
  private getMetadataInDecorator<
    K extends MergeKey<M, AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>
  >(metaKey: K, target: any, key?: any): GM[K] {
    const targetClass = target.constructor;
    if (key) {
      return Reflect.getMetadata(metaKey, targetClass, key);
    } else {
      return Reflect.getMetadata(metaKey, targetClass);
    }
  }

  private setMetadataInDecorator<
    K extends MergeKey<M, AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>
  >(metaKey: K, value: GM[K], target: any, key?: any) {
    const targetClass = target.constructor;
    if (key) {
      return Reflect.defineMetadata(metaKey, value, targetClass, key);
    } else {
      return Reflect.defineMetadata(metaKey, value, targetClass);
    }
  }

  transform<
    K extends MergeKey<M, AM>,
    IK extends Key<AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>
  >(
    metadataKey: K,
    metadataValueFun: (oldValue: GM[K]) => GM[K],
    keysIndexMeta?: IK,
  ): AllDecorators {
    return (target: any, key?: any, descriptor?: any) => {
      const oldValue = this.getMetadataInDecorator<K, GM>(
        metadataKey,
        target,
        key,
      );
      const newValue = metadataValueFun(oldValue);
      this.setMetadataInDecorator(metadataKey, newValue, target, key);
      if (keysIndexMeta) {
        const keysDec = this.append<IK, any>(keysIndexMeta, key);
        keysDec(target);
      }
      if (descriptor) {
        return descriptor;
      }
      return target;
    };
  }

  set<K extends MergeKey<M, AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: GenericMap<M, AM>[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK>(
      metadataKey,
      () => metadataValue,
      keysIndexMeta,
    );
  }

  append<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: AM[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => {
        const newArr = arr || [];
        newArr.push(metadataValue);
        return newArr;
      },
      keysIndexMeta,
    );
  }

  appendUnique<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: AM[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => {
        const newArr = arr || [];
        if (newArr.includes(metadataValue)) {
          return newArr;
        }
        newArr.push(metadataValue);
        return newArr;
      },
      keysIndexMeta,
    );
  }

  concat<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: ArrayValueMap<AM>[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => (arr || []).concat(metadataValue),
      keysIndexMeta,
    );
  }
}
