import 'reflect-metadata';
import { ArrayValueMap, GenericMap, Key, MergeKey, StringDict } from './def';
import { getClass } from './utility/utility';

export type AllDecorators = MethodDecorator &
  ClassDecorator &
  PropertyDecorator;

export class MetadataSetter<M extends StringDict, AM extends StringDict> {
  private getMetadataInDecorator<
    K extends MergeKey<M, AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>,
  >(metaKey: K, target: any, key?: any): GM[K] {
    if (key) {
      return Reflect.getMetadata(metaKey, target, key);
    } else {
      return Reflect.getMetadata(metaKey, target);
    }
  }

  private setMetadataInDecorator<
    K extends MergeKey<M, AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>,
  >(metaKey: K, value: GM[K], target: any, key?: any) {
    if (key) {
      return Reflect.defineMetadata(metaKey, value, target, key);
    } else {
      return Reflect.defineMetadata(metaKey, value, target);
    }
  }

  transform<
    K extends MergeKey<M, AM>,
    IK extends Key<AM>,
    GM extends Record<MergeKey<M, AM>, any> = GenericMap<M, AM>,
  >(
    metadataKey: K,
    metadataValueFun: (oldValue: GM[K]) => GM[K],
    keysIndexMeta?: IK,
  ): AllDecorators {
    return (target, key?) => {
      const targetClass = getClass(target);
      const oldValue = this.getMetadataInDecorator<K, GM>(
        metadataKey,
        targetClass,
        key,
      );
      const newValue = metadataValueFun(oldValue);
      this.setMetadataInDecorator(metadataKey, newValue, targetClass, key);
      if (keysIndexMeta) {
        const keysDec = this.appendUnique<IK, any>(keysIndexMeta, key);
        keysDec(targetClass);
      }
    };
  }

  set<K extends MergeKey<M, AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: GenericMap<M, AM>[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK>(
      metadataKey,
      () => metadataValue,
      keysIndexMeta,
    );
  }

  append<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: AM[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => {
        const newArr = arr || [];
        newArr.push(metadataValue);
        return newArr;
      },
      keysIndexMeta,
    );
  }

  appendUnique<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: AM[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => {
        const newArr = arr || [];
        if (newArr.includes(metadataValue)) {
          return newArr;
        }
        newArr.push(metadataValue);
        return newArr;
      },
      keysIndexMeta,
    );
  }

  concat<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: ArrayValueMap<AM>[K],
    keysIndexMeta?: IK,
  ) {
    return this.transform<K, IK, ArrayValueMap<AM>>(
      metadataKey,
      (arr) => (arr || []).concat(metadataValue),
      keysIndexMeta,
    );
  }

  param<K extends Key<AM>, IK extends Key<AM>>(
    metadataKey: K,
    metadataValue: AM[K],
    keysIndexMeta?: IK,
  ): ParameterDecorator {
    return (obj, key, i) => {
      const dec = this.transform<K, IK, ArrayValueMap<AM>>(
        metadataKey,
        (arr) => {
          const newArr = arr || [];
          newArr[i] = metadataValue;
          return newArr;
        },
        keysIndexMeta,
      );
      dec(obj, key);
    };
  }
}
