import { MetadataSetter, Reflector } from '../index';

describe('Reflector', () => {
  interface MetadataMap {
    foo: string;
  }

  interface MetadataArrayMap {
    bar: number;
    keys: string;
  }

  const Metadata = new MetadataSetter<MetadataMap, MetadataArrayMap>();
  const reflector = new Reflector<MetadataMap, MetadataArrayMap>();

  it('should set class metadata', () => {
    @Metadata.set('foo', 'first')
    class A {}

    class Noop {}

    const a = new A();
    const noop = new Noop();
    expect(reflector.get('foo', A)).toEqual('first');
    expect(reflector.get('foo', a)).toEqual('first');
    expect(reflector.get('foo', noop)).toBeUndefined();
    expect(reflector.get('foo', Noop)).toBeUndefined();
    expect(reflector.get('foo', Object)).toBeUndefined();
    expect(reflector.get('foo', Function)).toBeUndefined();
  });

  it('should set property metadata', () => {
    @Metadata.set('foo', 'first')
    class A {
      @Metadata.set('foo', 'second')
      public b: string;
    }

    class Noop {
      public b: string;
    }

    const a = new A();
    const noop = new Noop();
    expect(reflector.get('foo', A)).toEqual('first');
    expect(reflector.get('foo', a, 'b')).toEqual('second');
    expect(reflector.get('foo', A, 'b')).toEqual('second');
    expect(reflector.get('foo', Noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', Object, 'b')).toBeUndefined();
    expect(reflector.get('foo', Function, 'b')).toBeUndefined();
  });

  it('should set method metadata', () => {
    @Metadata.set('foo', 'first')
    class A {
      @Metadata.set('foo', 'second')
      public b() {}
    }

    class Noop {
      public b() {}
    }

    const a = new A();
    const noop = new Noop();
    expect(reflector.get('foo', A)).toEqual('first');
    expect(reflector.get('foo', a, 'b')).toEqual('second');
    expect(reflector.get('foo', A, 'b')).toEqual('second');
    expect(reflector.get('foo', Noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', Object, 'b')).toBeUndefined();
    expect(reflector.get('foo', Function, 'b')).toBeUndefined();
  });

  it('should set metadata in static fields', () => {
    @Metadata.set('foo', 'first')
    class A {
      @Metadata.set('foo', 'second')
      static b() {}

      @Metadata.set('foo', 'third')
      static c: string;
    }

    class Noop {
      static b() {}
      static c: string;
    }

    const a = new A();
    expect(reflector.get('foo', A)).toEqual('first');
    expect(reflector.get('foo', A, 'b')).toEqual('second');
    expect(reflector.get('foo', a, 'b')).toEqual('second');
    expect(reflector.get('foo', A, 'c')).toEqual('third');
    expect(reflector.get('foo', a, 'c')).toEqual('third');
    expect(reflector.get('foo', Noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', Noop, 'c')).toBeUndefined();
    expect(reflector.get('foo', Object, 'b')).toBeUndefined();
    expect(reflector.get('foo', Object, 'c')).toBeUndefined();
    expect(reflector.get('foo', Function, 'b')).toBeUndefined();
    expect(reflector.get('foo', Function, 'c')).toBeUndefined();
  });

  it('should set index keys metadata', () => {
    class A {
      @Metadata.set('foo', 'first', 'keys')
      foo: string;

      @Metadata.set('foo', 'second', 'keys')
      bar: string;
    }

    const a = new A();
    expect(reflector.get('foo', a, 'foo')).toEqual('first');
    expect(reflector.get('foo', A, 'foo')).toEqual('first');
    expect(reflector.get('foo', a, 'bar')).toEqual('second');
    expect(reflector.get('foo', A, 'bar')).toEqual('second');
    expect(reflector.get('keys', A)).toEqual(['foo', 'bar']);
    expect(reflector.get('keys', a)).toEqual(['foo', 'bar']);
  });

  it('should set metadata on extended class', () => {
    @Metadata.set('foo', 'first')
    class A {
      @Metadata.set('foo', 'second')
      public b: string;
      @Metadata.set('foo', 'fourth')
      public c: string;
    }

    class B extends A {
      @Metadata.set('foo', 'third')
      public b: string;
      public c: string;
    }

    @Metadata.set('foo', 'fifth')
    class C extends A {}

    class Noop {
      public b: string;
    }

    const a = new A();
    const b = new B();
    const c = new C();
    const noop = new Noop();
    expect(reflector.get('foo', A)).toEqual('first');
    expect(reflector.get('foo', a)).toEqual('first');
    expect(reflector.get('foo', B)).toEqual('first');
    expect(reflector.get('foo', b)).toEqual('first');
    expect(reflector.get('foo', C)).toEqual('fifth');
    expect(reflector.get('foo', c)).toEqual('fifth');
    expect(reflector.get('foo', a, 'b')).toEqual('second');
    expect(reflector.get('foo', A, 'b')).toEqual('second');
    expect(reflector.get('foo', B, 'b')).toEqual('third');
    expect(reflector.get('foo', b, 'b')).toEqual('third');
    expect(reflector.get('foo', B, 'c')).toEqual('fourth');
    expect(reflector.get('foo', b, 'c')).toEqual('fourth');
    expect(reflector.get('foo', Noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', noop, 'b')).toBeUndefined();
    expect(reflector.get('foo', Object, 'b')).toBeUndefined();
    expect(reflector.get('foo', Function, 'b')).toBeUndefined();
  });

  it('should inherit the keys on inherited class', () => {
    class A {
      @Metadata.set('foo', 'first', 'keys')
      public a: string;
    }

    class AMid extends A {}

    class B extends AMid {
      @Metadata.set('foo', 'second', 'keys')
      public b: string;
    }

    const a = new A();
    const b = new B();

    expect(reflector.get('keys', A)).toEqual(['a', 'b']);
    expect(reflector.get('keys', a)).toEqual(['a', 'b']);
    expect(reflector.get('keys', B)).toEqual(['a', 'b']);
    expect(reflector.get('keys', b)).toEqual(['a', 'b']);
  });

  it('should work with parameter decorators', () => {
    class A {
      constructor(
        @Metadata.param('bar', 1) a: string,
        @Metadata.param('bar', 2) b: string,
      ) {}

      method(
        @Metadata.param('bar', 3) a: string,
        @Metadata.param('bar', 4) b: string,
      ) {}

      static staticMethod(
        @Metadata.param('bar', 5) a: string,
        @Metadata.param('bar', 6) b: string,
      ) {}
    }

    const a = new A('foo', 'bar');
    expect(reflector.get('bar', A)).toStrictEqual([1, 2]);
    expect(reflector.get('bar', a)).toStrictEqual([1, 2]);
    expect(reflector.get('bar', A, 'method')).toStrictEqual([3, 4]);
    expect(reflector.get('bar', a, 'method')).toStrictEqual([3, 4]);
    expect(reflector.get('bar', A, 'staticMethod')).toStrictEqual([5, 6]);
    expect(reflector.get('bar', a, 'staticMethod')).toStrictEqual([5, 6]);
  });
});
