import { RegisterSchema, SchemaProperty } from '../src/decorators';
import Schema from 'schemastery';

describe('Schema dict', () => {
  @RegisterSchema()
  class MyProperty {
    @SchemaProperty({ default: 'default' })
    myName: string;

    getName() {
      return this.myName;
    }
  }

  interface FooAndBar {
    foo: number;
    bar: number;
  }

  @RegisterSchema()
  class MyConfig {
    constructor(_: any) {}

    @SchemaProperty({ type: String, dict: true })
    foo: Record<string, string>;

    @SchemaProperty({ type: Number, dict: Schema.union(['foo', 'bar']) })
    fooAndBar: FooAndBar;

    @SchemaProperty({ dict: true, type: MyProperty, default: {} })
    myProperties: Record<string, MyProperty>;

    @SchemaProperty({ dict: true, type: MyProperty, default: [] })
    myPropertiesArr: Record<string, MyProperty>[];
  }

  const schema = MyConfig as Schema<MyConfig>;

  function checkSchema(schema: Schema) {
    expect(schema.dict.foo.type).toEqual('dict');
    expect(schema.dict.fooAndBar.type).toEqual('dict');
    expect(schema.dict.myProperties.type).toEqual('dict');
    expect(schema.dict.myPropertiesArr.type).toEqual('array');
  }

  it('should be correct type', () => {
    checkSchema(schema);
  });

  it('should serialize', () => {
    const data = schema.toJSON();
    console.log(JSON.stringify(data, null, 2));
    const restoredSchema = new Schema(data);
    checkSchema(restoredSchema);
  });

  it('should check dict', () => {
    const config = new MyConfig({
      foo: {
        bar: 'baz',
      },
      fooAndBar: {
        foo: 1,
        bar: 2,
      },
    });
    expect(config.foo).toEqual({ bar: 'baz' });
    expect(config.fooAndBar).toEqual({ foo: 1, bar: 2 });
  });

  it('should throw on invalid', () => {
    expect(() => new MyConfig({ foo: 1 })).toThrow();
    expect(() => new MyConfig({ foo: { bar: 1 } })).toThrow();
    expect(() => new MyConfig({ fooAndBar: { foo: 'not number' } })).toThrow();
    expect(() => new MyConfig({ fooAndBar: { somethingElse: 4 } })).toThrow();
  });

  it('should be instance of MyProperty', () => {
    const config = new MyConfig({
      myProperties: {
        foo: {
          myName: 'foo',
        },
      },
      myPropertiesArr: [
        {
          bar: {
            myName: 'bar',
          },
        },
      ],
    });
    expect(config.myProperties.foo.getName()).toEqual('foo');
    expect(config.myPropertiesArr[0].bar.getName()).toEqual('bar');
  });
});
