import { RegisterSchema, SchemaProperty } from '../src/decorators';
import { kSchema } from '../src/utility/kschema';
import Schema from 'schemastery';

describe('Schema registration', () => {
  @RegisterSchema()
  class Config {
    constructor(_config: any) {}

    @SchemaProperty({ default: 'bar' })
    foo: string;
  }

  @RegisterSchema()
  class ConfigWithRequiredFields {
    constructor(_config: any) {}

    @SchemaProperty({ required: true })
    foo: string;
  }

  it('should be a schema', () => {
    expect(Config[kSchema]).toBeDefined();
  });

  it('should be schema object type', () => {
    const schema = Config as Schema<Config>;
    expect(schema.type).toBe('object');
    expect(schema.dict.foo.type).toBe('string');
  });

  it('should be serializable', () => {
    const schema = Config as Schema<Config>;
    const plainObject: Partial<Schema<Config>> = JSON.parse(
      JSON.stringify(schema),
    );
    expect(plainObject.type).toBe('object');
    expect(plainObject.dict.foo.type).toBe('string');
  });

  it('should put default value', () => {
    expect(new Config({}).foo).toBe('bar');
    expect(new Config({ foo: 'baz' }).foo).toBe('baz');
  });

  it('should throw if no default value', () => {
    expect(() => new Config({ foo: 111 })).toThrow();
  });

  it('should throw if required field is missing', () => {
    expect(() => new ConfigWithRequiredFields({})).toThrow();
  });

  it('should act functions of schema', () => {
    const schema = Config as Schema<Config>;
    const newSchema = schema.description('foo');
    expect(newSchema.meta.description).toBe('foo');
  });

  it('should work in toString method', () => {
    const schema = Config as Schema<Config>;
    expect(schema.toString(true)).toBe('{ foo?: string }');
  });
});
