import {
  ClassType,
  GeneratedSym,
  RefSym,
  SchemaClassOptions,
  SchemaOptions,
  SchemaOptionsDict,
  SchemaOrReference,
  SchemaReference,
  SchemaType,
} from '../def';
import Schema from 'schemastery';
import { reflector } from '../metadata/reflector';
import _ from 'lodash';
import { Metadata } from '../metadata/metadata';
import { kSchema } from '../utility/kschema';

export function resolveSchemaType(schemaType: SchemaType): SchemaOrReference {
  if (schemaType && schemaType[RefSym]) {
    return schemaType as SchemaReference;
  }
  // eslint-disable-next-line @typescript-eslint/ban-types
  switch (schemaType as string | Function) {
    case 'any':
      return Schema.any();
    case 'never':
      return Schema.never();
    case 'string':
    case String:
      return Schema.string();
    case 'number':
    case Number:
      return Schema.number();
    case 'boolean':
    case Boolean:
      return Schema.boolean();
    case 'object':
      return Schema.object({}).default({});
    case Function:
      return Schema.function();
    default:
      return Schema.from(schemaType);
  }
}

function applyOptionsToSchema(schema: Schema, options: SchemaClassOptions) {
  Object.assign(schema.meta, {
    ...options,
    type: undefined,
    dict: undefined,
    array: undefined,
  });
}

function getPropertySchemaFromOptions<PT>(
  options: SchemaOptions,
): SchemaOrReference<PT> {
  const _schema = resolveSchemaType(options.type);
  if (_schema[RefSym]) {
    return _schema as SchemaReference<PT>;
  }
  let schema = _schema as Schema;
  if (options.dict) {
    const skeySchema = (
      options.dict === true
        ? Schema.string()
        : (Schema.from(options.dict) as unknown as Schema<any, string>)
    ) as Schema<any, string>;
    schema = Schema.dict(schema, skeySchema).default({});
  }
  if (options.array) {
    schema = Schema.array(schema).default([]);
  }
  applyOptionsToSchema(schema, options);
  return schema;
}

function resolveSchemaReference<S = any, T = S>(ref: SchemaReference<S, T>) {
  const value = ref.factory();
  /*if (value[kSchema]) {
    return value;
  }
  if (typeof value === 'function' && !value[kSchema]) {
    return SchemaClass(value as { new (...args: any[]): any });
  }*/
  return value;
}

function resolvePropertySchemaFromOptions<PT>(
  options: SchemaOptions,
): Schema<PT> {
  const schema = getPropertySchemaFromOptions<PT>(options);
  if (schema[RefSym]) {
    return resolvePropertySchemaFromOptions({
      ...options,
      type: resolveSchemaReference(schema as SchemaReference),
    });
  }
  return schema as Schema<PT>;
}

function tempDefGet<T, K extends keyof T>(obj: T, key: K, getter: () => T[K]) {
  Object.defineProperty(obj, key, {
    get: getter,
    set: (value: T[K]) => {
      Object.defineProperty(obj, key, {
        enumerable: true,
        configurable: true,
        writable: true,
        value,
      });
    },
    enumerable: true,
    configurable: true,
  });
}

function schemasFromDict<T>(dict: SchemaOptionsDict<T>) {
  const schemaDict: {
    [P in keyof SchemaOptionsDict<T>]: SchemaOrReference<
      SchemaOptionsDict<T>[P]
    >;
  } = _.mapValues(dict, (opt) => getPropertySchemaFromOptions(opt));
  const schema = Schema.object({});
  for (const _key of Object.keys(schemaDict)) {
    const key = _key as keyof T;
    if (schemaDict[key][RefSym]) {
      const schemaOptions = dict[key];
      tempDefGet(schema.dict, _key, () =>
        resolvePropertySchemaFromOptions(schemaOptions),
      );
    } else {
      schema.dict[_key] = schemaDict[key] as Schema<T[keyof T]>;
    }
  }
  return schema;
}

function schemaOptionsFromClass<T>(cl: ClassType<T>): SchemaOptionsDict<T> {
  const keys = reflector.getArray('SchemaMetaKey', cl) as (keyof T &
    (string | symbol))[];
  if (!keys) {
    return null;
  }
  const result: SchemaOptionsDict<T> = {};
  for (const key of keys) {
    const option = reflector.get('SchemaMeta', cl, key);
    if (!option) {
      continue;
    }
    result[key] = option;
  }
  return result;
}

export function schemaFromClass<T>(cl: ClassType<T>): Schema<Partial<T>, T> {
  let schema: Schema;
  const optionsDict = schemaOptionsFromClass(cl);
  if (!optionsDict) {
    schema = Schema.object({});
  } else {
    schema = schemasFromDict<T>(optionsDict);
  }
  const classOptions = reflector.get('SchemaClassOptions', cl);
  if (classOptions) {
    applyOptionsToSchema(schema, classOptions);
  }
  return schema;
}

const schemaFields: (keyof Schema)[] = [
  'type',
  'sKey',
  'inner',
  'list',
  'dict',
  'callback',
  'value',
  'meta',
  'uid',
  'refs',
];

const schemaFunctions: (keyof Schema)[] = [
  // 'toJSON',
  'required',
  'hidden',
  'role',
  'link',
  'default',
  'comment',
  'description',
  'max',
  'min',
  'step',
  'set',
  'push',
];

function applySchemaForClass<T>(
  schema: Schema<Partial<T>, T>,
  originalClass: ClassType<T>,
  instance: T,
  originalObject: Partial<T>,
) {
  const newRawObject = new schema(originalObject);
  for (const key in schema.dict) {
    const transformer = reflector.get('Transformer', originalClass, key);
    if (transformer) {
      newRawObject[key] = transformer(newRawObject[key]);
    }
  }
  for (const key in newRawObject) {
    Object.defineProperty(instance, key, {
      writable: true,
      enumerable: true,
      configurable: true,
      value: newRawObject[key],
    });
  }
  return instance;
}

export function SchemaClass<T>(originalClass: ClassType<T>) {
  const schema = schemaFromClass(originalClass);
  const newClass = function (...args: any[]): T {
    const instance = new originalClass(...args);
    const originalObject = args[0];
    return applySchemaForClass(schema, originalClass, instance, originalObject);
  } as unknown as ClassType<T> & Schema<Partial<T>, T>;
  newClass[GeneratedSym] = schema;
  Object.defineProperty(newClass, 'name', {
    value: originalClass.name,
  });
  Object.setPrototypeOf(newClass, originalClass.prototype);
  for (const field of schemaFields) {
    Object.defineProperty(newClass, field, {
      configurable: true,
      enumerable: true,
      get() {
        return newClass[GeneratedSym][field];
      },
      set(value) {
        newClass[GeneratedSym][field] = value;
      },
    });
  }
  for (const functionField of schemaFunctions) {
    if (newClass[functionField]) {
      continue;
    }
    Object.defineProperty(newClass, functionField, {
      configurable: true,
      enumerable: false,
      writable: false,
      value: Schema.prototype[functionField].bind(newClass),
    });
  }

  Object.defineProperty(newClass, 'toJSON', {
    configurable: true,
    enumerable: false,
    writable: false,
    value: () =>
      Schema.prototype.toJSON.call({ ...newClass, toJSON: undefined }),
  });

  newClass.toString = schema.toString.bind(schema);
  newClass[kSchema] = true;
  return newClass;
}

export function RegisterSchema(options: SchemaClassOptions = {}) {
  return function <T>(originalClass: ClassType<T>): any {
    Metadata.set('SchemaClassOptions', options)(originalClass);
    return SchemaClass(originalClass);
  };
}
