import { DefineSchema, RegisterSchema, SchemaRef } from '../src/decorators';
import { kSchema } from '../src/utility/kschema';
import Schema from 'schemastery';

@RegisterSchema()
class Author {
  constructor(_: Partial<Author>) {}

  @DefineSchema()
  name: string;

  getName?() {
    return this.name;
  }

  @DefineSchema({
    type: SchemaRef(() => Post),
  })
  posts?: Post[];
}

@RegisterSchema()
class Post {
  constructor(_: Partial<Post>) {}

  @DefineSchema()
  name: string;

  getName?() {
    return this.name;
  }

  @DefineSchema({
    type: SchemaRef(() => Author),
  })
  author?: Author;

  @DefineSchema({
    type: SchemaRef(() => Post),
  })
  childPosts?: Post[];
}

describe('Recurse schema', () => {
  it('should be a Schema', () => {
    expect(Post[kSchema]).toBeDefined();
  });

  it('child should be Schema', () => {
    const authorSchema = (Post as Schema<Author>).dict.author;
    expect(authorSchema[kSchema]).toBeDefined();
  });

  it('child of child should be Schema', () => {
    const authorSchema = (Post as Schema<Post>).dict.childPosts.inner.dict
      .author;
    expect(authorSchema[kSchema]).toBeDefined();
  });

  it('should resolve value', () => {
    const post = new Post({
      name: 'test',
    });
    expect(post.getName()).toBe('test');
  });

  it('should be able to recurse', () => {
    const post = new Post({
      name: 'test',
      author: {
        name: 'Shigma',
      },
      childPosts: [{ name: 'child' }],
    });

    expect(post.author.getName()).toBe('Shigma');
    expect(post.childPosts[0].getName()).toBe('child');
  });

  it('should recurse twice', () => {
    const post = new Post({
      name: 'test',
      author: {
        name: 'Shigma',
        posts: [{ name: 'test' }, { name: 'test1' }],
      },
    });

    expect(post.getName()).toBe('test');
    expect(post.author.getName()).toBe('Shigma');
    expect(post.author.posts[1].getName()).toBe('test1');
  });
});
