import {
  ClassType,
  SchemaClassOptions,
  SchemaOptions,
  SchemaOptionsDict,
} from '../def';
import Schema from 'schemastery';
import { reflector } from '../metadata/reflector';
import _ from 'lodash';
import { Metadata } from '../metadata/metadata';

function getBasePropertySchemaFromOptions(options: SchemaOptions) {
  if (options.schema) {
    return options.schema;
  }
  if (typeof options.type !== 'string') {
    return schemaFromClass(options.type);
  }
  switch (options.type as string) {
    case 'any':
      return Schema.any();
    case 'never':
      return Schema.never();
    case 'string':
      return Schema.string();
    case 'number':
      return Schema.number();
    case 'boolean':
      return Schema.boolean();
    case 'object':
      return Schema.object({}, true);
    default:
      return Schema.any();
  }
}

function applyOptionsToSchema(schema: Schema, options: SchemaClassOptions) {
  if (options.required != undefined) {
    schema.meta.required = options.required;
  }
  if (options.hidden != undefined) {
    schema.meta.hidden = options.hidden;
  }
  if (options.default != undefined) {
    schema.meta.default = options.default;
  }
  if (options.comment != undefined) {
    schema.meta.comment = options.comment;
  }
  if (options.allowUnknown != undefined) {
    schema.flag = options.allowUnknown;
  }
  if (options.desc != undefined) {
    schema.desc = options.desc;
  }
}

function getPropertySchemaFromOptions<PT>(options: SchemaOptions): Schema<PT> {
  let schema = getBasePropertySchemaFromOptions(options);
  if (options.dict) {
    schema = Schema.dict(schema);
  }
  if (options.array) {
    schema = Schema.array(schema);
  }
  applyOptionsToSchema(schema, options);
  return schema;
}

function schemasFromDict<T>(dict: SchemaOptionsDict<T>): Schema<T> {
  // eslint-disable-next-line @typescript-eslint/ban-ts-comment
  // @ts-ignore
  return Schema.object<T>(
    _.mapValues(dict, (opt) => getPropertySchemaFromOptions(opt)),
  );
}

function schemaOptionsFromClass<T>(cl: ClassType<T>): SchemaOptionsDict<T> {
  const keys = reflector.getArray('SchemaMetaKey', cl) as (keyof T &
    (string | symbol))[];
  if (!keys) {
    return null;
  }
  const result: SchemaOptionsDict<T> = {};
  for (const key of keys) {
    const option = reflector.get('SchemaMeta', cl, key);
    result[key] = option;
  }
  return result;
}

export function schemaFromClass<T>(cl: ClassType<T>): Schema<T> {
  let schema: Schema;
  const optionsDict = schemaOptionsFromClass(cl);
  if (!optionsDict) {
    schema = Schema.object({}, true);
  } else {
    schema = schemasFromDict<T>(optionsDict);
  }
  const classOptions = reflector.get('SchemaClassOptions', cl);
  if (classOptions) {
    applyOptionsToSchema(schema, classOptions);
  }
  return schema;
}

const schemaFields: (keyof Schema.Base)[] = [
  'type',
  'desc',
  'key',
  'flag',
  'value',
  'alt',
  'sDict',
  'list',
  'dict',
  'callback',
  'meta',
];

function applySchemaForClass<T>(
  originalClass: ClassType<T>,
  instance: T,
  originalObject: any,
) {
  const schema = schemaFromClass(originalClass);
  const newRawObject = new schema(originalObject);
  for (const key in schema.dict) {
    const transformer = reflector.get('Transformer', originalClass, key);
    if (transformer) {
      newRawObject[key] = transformer(newRawObject[key]);
    }
  }
  for (const key in newRawObject) {
    Object.defineProperty(instance, key, {
      writable: true,
      enumerable: true,
      configurable: true,
      value: newRawObject[key],
    });
  }
  return instance;
}

export function SchemaClass<T>(originalClass: ClassType<T>) {
  const schema = schemaFromClass(originalClass);
  const newClass = (function (...args: any[]): T {
    const instance = new originalClass(...args);
    const originalObject = args[0];
    return applySchemaForClass(originalClass, instance, originalObject);
  } as unknown) as ClassType<T> & Schema<T>;
  Object.defineProperty(newClass, 'name', {
    value: originalClass.name,
  });
  Object.setPrototypeOf(newClass, originalClass.prototype);
  for (const field of schemaFields) {
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    newClass[field] = schema[field];
  }
  return newClass;
}

export function RegisterSchema(options: SchemaClassOptions = {}) {
  return function <T>(originalClass: ClassType<T>): any {
    Metadata.set('SchemaClassOptions', options)(originalClass);
    return SchemaClass(originalClass);
  };
}

// for backward compatibility
export const SchemaConf = RegisterSchema;
