import {
  ClassType,
  SchemaClassOptions,
  SchemaOptions,
  SchemaOptionsDict,
} from '../def';
import Schema from 'schemastery';
import { reflector } from '../metadata/reflector';
import _ from 'lodash';
import { Metadata } from '../metadata/metadata';

function getBasePropertySchemaFromOptions(options: SchemaOptions) {
  if (options.schema) {
    return options.schema;
  }
  if (typeof options.type !== 'string') {
    return schemaFromClass(options.type);
  }
  switch (options.type as string) {
    case 'any':
      return Schema.any();
    case 'never':
      return Schema.never();
    case 'string':
      return Schema.string();
    case 'number':
      return Schema.number();
    case 'boolean':
      return Schema.boolean();
    case 'object':
      return Schema.object({}).default({});
    default:
      return Schema.any();
  }
}

function applyOptionsToSchema(schema: Schema, options: SchemaClassOptions) {
  if (options.desc) {
    schema.meta.description = options.desc;
  }
  Object.assign(schema.meta, options);
}

function getPropertySchemaFromOptions<PT>(options: SchemaOptions): Schema<PT> {
  let schema = getBasePropertySchemaFromOptions(options);
  if (options.dict) {
    schema = Schema.dict(schema).default({});
  }
  if (options.array) {
    schema = Schema.array(schema).default([]);
  }
  applyOptionsToSchema(schema, options);
  return schema;
}

function schemasFromDict<T>(dict: SchemaOptionsDict<T>) {
  const schemaDict: {
    [P in keyof SchemaOptionsDict<T>]: Schema<SchemaOptionsDict<T>[P]>;
  } = _.mapValues(dict, (opt) => getPropertySchemaFromOptions(opt));
  return Schema.object(schemaDict);
}

function schemaOptionsFromClass<T>(cl: ClassType<T>): SchemaOptionsDict<T> {
  const keys = reflector.getArray('SchemaMetaKey', cl) as (keyof T &
    (string | symbol))[];
  if (!keys) {
    return null;
  }
  const result: SchemaOptionsDict<T> = {};
  for (const key of keys) {
    const option = reflector.get('SchemaMeta', cl, key);
    result[key] = option;
  }
  return result;
}

export function schemaFromClass<T>(cl: ClassType<T>): Schema<Partial<T>, T> {
  let schema: Schema;
  const optionsDict = schemaOptionsFromClass(cl);
  if (!optionsDict) {
    schema = Schema.object({});
  } else {
    schema = schemasFromDict<T>(optionsDict);
  }
  const classOptions = reflector.get('SchemaClassOptions', cl);
  if (classOptions) {
    applyOptionsToSchema(schema, classOptions);
  }
  return schema;
}

const schemaFields: (keyof Schema.Base)[] = [
  'type',
  'inner',
  'list',
  'dict',
  'callback',
  'value',
  'meta',
];

function applySchemaForClass<T>(
  originalClass: ClassType<T>,
  instance: T,
  originalObject: Partial<T>,
) {
  const schema = schemaFromClass(originalClass);
  const newRawObject = new schema(originalObject);
  for (const key in schema.dict) {
    const transformer = reflector.get('Transformer', originalClass, key);
    if (transformer) {
      newRawObject[key] = transformer(newRawObject[key]);
    }
  }
  for (const key in newRawObject) {
    Object.defineProperty(instance, key, {
      writable: true,
      enumerable: true,
      configurable: true,
      value: newRawObject[key],
    });
  }
  return instance;
}

export function SchemaClass<T>(originalClass: ClassType<T>) {
  const schema = schemaFromClass(originalClass);
  const newClass = (function (...args: any[]): T {
    const instance = new originalClass(...args);
    const originalObject = args[0];
    return applySchemaForClass(originalClass, instance, originalObject);
  } as unknown) as ClassType<T> & Schema<Partial<T>, T>;
  Object.defineProperty(newClass, 'name', {
    value: originalClass.name,
  });
  Object.setPrototypeOf(newClass, originalClass.prototype);
  for (const field of schemaFields) {
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    newClass[field] = schema[field];
  }
  return newClass;
}

export function RegisterSchema(options: SchemaClassOptions = {}) {
  return function <T>(originalClass: ClassType<T>): any {
    Metadata.set('SchemaClassOptions', options)(originalClass);
    return SchemaClass(originalClass);
  };
}

// for backward compatibility
export const SchemaConf = RegisterSchema;
