import { Aragami, ClassType } from 'aragami';
import { ConsoleLogger, Logger } from '@nestjs/common';
import * as os from 'os';
import { InjectAragami } from '../index';

export interface QueueRunnerOptions {
  maxConcurrency?: number;
  key?: string;
  logTask?: boolean;
  retry?: number;
}

export class _QueueRunner<T> {
  logger = new Logger(this._queueClass.name);
  _maxConcurrency = this._queueRunnerOptions.maxConcurrency ?? os.cpus().length;

  constructor(
    public _queueClass: ClassType<T>,
    public _queueRunnerOptions: QueueRunnerOptions,
  ) {}

  @InjectAragami()
  aragami: Aragami;

  async onApplicationBootstrap() {
    await this.aragami.queueResumeAll(
      this._queueClass,
      this._queueRunnerOptions.key,
      true,
    );
    if (this._queueRunnerOptions.logTask) {
      this.logger.log(`Starting ${this._maxConcurrency} workers`);
    }
    for (let i = 0; i < this._maxConcurrency; i++) {
      if (this._queueRunnerOptions.logTask) {
        this.logger.log(`Starting worker ${i}`);
      }
      this._mainLoop(i).then();
    }
  }

  _quit = false;

  async onApplicationShutdown() {
    this._quit = true;
  }

  async runTask(task: T, i: number) {
    return;
  }

  _queueKey = this._queueRunnerOptions.key || 'default';

  async _mainLoop(i: number) {
    const logger = new ConsoleLogger(`${this._queueClass.name}-${i}`);
    while (!this._quit) {
      logger.log('Looping');
      try {
        await this.aragami.runQueueOnce(
          this._queueClass,
          async (task) => {
            if (this._queueRunnerOptions.logTask) {
              logger.log(`Got task: ${JSON.stringify(task)}`);
            }
            try {
              await this.runTask(task, i);
            } catch (e) {
              logger.error(`Task failed: ${e}`);
              if (
                this._queueRunnerOptions.retry &&
                (!task['__retry'] ||
                  task['__retry'] < this._queueRunnerOptions.retry)
              ) {
                logger.log(`Retrying task: ${JSON.stringify(task)}`);
                if (this._queueRunnerOptions.retry > 0) {
                  task['__retry'] = (task['__retry'] || 0) + 1;
                }
                await this.aragami.queueAdd(task, {
                  key: this._queueKey,
                  prior: true,
                });
              }
            }
          },
          this._queueKey,
        );
        //
      } catch (e) {
        logger.error(`Loop failed: ${e}`);
      }
    }
  }
}

export function QueueRunner<T>(
  queueClass: ClassType<T>,
  options?: Partial<QueueRunnerOptions>,
) {
  const cl = class extends _QueueRunner<T> {
    constructor(innerOptions?: Partial<QueueRunnerOptions>) {
      super(queueClass, {
        ...(innerOptions || {}),
        ...(options || {}),
      });
    }
  };
  Object.defineProperty(cl, 'name', { value: queueClass.name + 'Runner' });
  return cl;
}
