import {
  CallHandler,
  ExecutionContext,
  Inject,
  Injectable,
  NestInterceptor,
  Optional,
} from '@nestjs/common';
import { sortParams } from './utility/sort-params';
import { InjectAragami } from '../index';
import { Aragami, CacheKey } from 'aragami';
import { reflector } from 'aragami/dist/src/metadata';
import { of, tap } from 'rxjs';
import { HttpAdapterHost } from '@nestjs/core';

class RouteCache {
  @CacheKey()
  key: string;
  value: any;
}

@Injectable()
export class AragamiCacheInterceptor implements NestInterceptor {
  @Optional()
  @Inject()
  protected readonly httpAdapterHost: HttpAdapterHost;

  constructor(@InjectAragami() protected aragami: Aragami) {}

  protected allowedMethods = ['GET'];
  async intercept(context: ExecutionContext, next: CallHandler) {
    const key = await this.trackBy(context);
    if (!key) {
      return next.handle();
    }
    const controllerCls = context.getClass();
    const controllerMethodName = context.getHandler().name;
    const ttl =
      reflector.get('AragamiCacheTTL', controllerCls, controllerMethodName) ||
      reflector.get('AragamiCacheTTL', controllerCls);
    if (!ttl) {
      return next.handle();
    }
    const cachedValue = await this.aragami.get(RouteCache, key);
    if (cachedValue) {
      return of(cachedValue.value);
    }
    return next
      .handle()
      .pipe(
        tap((value) => this.aragami.set(RouteCache, { key, value }, { ttl })),
      );
  }

  protected async trackBy(
    context: ExecutionContext,
  ): Promise<string | undefined> {
    if (!(await this.isRequestCacheable(context))) {
      return;
    }
    const httpAdapter = this.httpAdapterHost.httpAdapter;
    const isHttpApp = httpAdapter && !!httpAdapter.getRequestMethod;
    if (!isHttpApp) {
      return;
    }
    const url = new URL(
      httpAdapter.getRequestUrl(context.getArgByIndex(0)),
      'http://localhost',
    );
    return `${url.pathname}?${sortParams(url.searchParams)}`;
  }

  protected async isRequestCacheable(
    context: ExecutionContext,
  ): Promise<boolean> {
    const req = context.switchToHttp().getRequest();
    return this.allowedMethods.includes(req.method);
  }
}
