import { Type } from 'class-transformer';
import 'reflect-metadata';
import moment from 'moment';
import _ from 'lodash';

export enum ReportScoreResult {
  NotFound = 'NotFound',
  Continue = 'Continue',
  NextRound = 'NextRound',
  Finish = 'Finish',
}

function generateSpaces(n: number) {
  return _.range(n)
    .map(() => ' ')
    .join('');
}

export class Match {
  scores: number[];
  winner: string;
  constructor(public players: string[]) {
    if (!players) {
      return;
    }
    this.scores = [0, 0];
  }
  foo() {
    return 'bar';
  }
  setScore(scores: number[]) {
    this.scores = scores;
    const [scorea, scoreb] = scores;
    if (scorea === scoreb) {
      return;
    } else if (scorea > scoreb) {
      this.winner = this.players[0];
    } else {
      this.winner = this.players[1];
    }
  }

  reportScore(name: string, self: number, oppo: number) {
    const pos = this.players.findIndex(
      (p) => name.startsWith(p) || name.endsWith(p),
    );
    if (pos === -1) {
      return false;
    }
    const score: number[] = [];
    score[pos] = self;
    score[1 - pos] = oppo;
    this.setScore(score);
    return true;
  }

  isFinished() {
    return this.winner || this.scores.some((s) => s !== 0);
  }

  format(leftMaxLength: number, rightMaxLength: number) {
    const leftSpace = leftMaxLength - this.players[0].length + 2;
    const rightSpace = rightMaxLength - this.players[1].length + 2;
    return `${this.players[0]}${generateSpaces(leftSpace)}${this.scores[0]}:${
      this.scores[1]
    }${generateSpaces(rightSpace)}${this.players[1]}`;
  }
}

function shuffle<T>(_arr: T[]): T[] {
  const arr = Array.from(_arr);
  for (let i = arr.length - 1; i >= 0; i--) {
    const rIndex = Math.floor(Math.random() * (i + 1));
    const temp = arr[rIndex];
    arr[rIndex] = arr[i];
    arr[i] = temp;
  }
  return arr;
}

export class Round {
  @Type(() => Match)
  matches: Match[];
  constructor(_playerLists: string[][]) {
    if (!_playerLists) {
      return;
    }
    const playerLists = _playerLists.map((plist) => shuffle(plist));
    const matchCount = Math.min(...playerLists.map((l) => l.length));
    this.matches = [];
    for (let i = 0; i < matchCount; ++i) {
      const [lista, listb] = playerLists;
      this.matches.push(new Match([lista[i], listb[i]]));
    }
  }

  reportScore(name: string, selfScore: number, oppoScore: number) {
    const match = this.matches.find((m) =>
      m.players.some((p) => name.startsWith(p) || name.endsWith(p)),
    );
    if (!match) {
      return;
    }
    return match.reportScore(name, selfScore, oppoScore);
  }

  isAllFinished() {
    return this.matches.every((m) => m.isFinished());
  }
  format(leftMaxLength: number, rightMaxLength: number) {
    return this.matches
      .map((m) => m.format(leftMaxLength, rightMaxLength))
      .join('\n');
  }
}

export class Game {
  @Type(() => Round)
  rounds: Round[];
  constructor(
    public location: string,
    public time: Date,
    public rule: string,
    public teama: string,
    public teamb: string,
    public playera: string[],
    public playerb: string[],
  ) {
    if (!teama) {
      return;
    }
    this.rounds = [];
    this.newRound();
  }

  newRound() {
    if (this.rule === '人头赛' && this.rounds.length) {
      return false;
    }
    const playerLists = [this.playera, this.playerb].map((plist) =>
      plist.filter((p) => this.isPlayerAvailable(p)),
    );
    if (playerLists.some((l) => !l.length)) {
      return false;
    }
    this.rounds.push(new Round(playerLists));
    return true;
  }

  isPlayerAvailable(name: string) {
    return !this.rounds.some((round) =>
      round.matches.some(
        (match) => match.players.includes(name) && match.winner !== name,
      ),
    );
  }

  reportScore(name: string, selfScore: number, oppoScore: number) {
    const round = this.rounds[this.rounds.length - 1];
    const result = round.reportScore(name, selfScore, oppoScore);
    if (!result) {
      return ReportScoreResult.NotFound;
    }
    if (round.isAllFinished()) {
      const nextRoundResult = this.newRound();
      if (nextRoundResult) {
        return ReportScoreResult.NextRound;
      } else {
        return ReportScoreResult.Finish;
      }
    } else {
      return ReportScoreResult.Continue;
    }
  }

  format() {
    const allMatches = _.flatten(this.rounds.map((r) => r.matches));
    const leftMaxLength = Math.max(
      ..._.map(allMatches, (m) => m.players[0].length),
    );
    const rightMaxLength = Math.max(
      ..._.map(allMatches, (m) => m.players[1].length),
    );

    const lines: string[] = [];
    lines.push(`战队： ${this.teama} VS ${this.teamb}`);
    lines.push(`时间： ${moment(this.time).format('YYYY-MM-DD')}`);
    lines.push(`规则： ${this.rule}`);
    lines.push(`地点： ${this.location}`);
    lines.push('------------第一轮------------');
    lines.push(this.rounds[0].format(leftMaxLength, rightMaxLength));
    lines.push('------------第二轮------------');
    lines.push(
      this.rounds
        .slice(1)
        .map((r) => r.format(leftMaxLength, rightMaxLength))
        .join('\n'),
    );
    return lines.join('\n');
  }
  getKey() {
    return `${this.location}_${this.teama}_${this.teamb}`;
  }

  includesPendingPlayer(possibleNames: string[]) {
    return possibleNames.find((name) =>
      this.rounds[this.rounds.length - 1].matches.some((match) =>
        match.players.some(
          (playerName) =>
            name.startsWith(playerName) || name.endsWith(playerName),
        ),
      ),
    );
  }
}
