import { Adapter, Bot, Context, Filter } from 'koishi';
import _ from 'lodash';
import { RegisterSchema, SchemaProperty } from 'schemastery-gen';

export interface ChannelTargetLike {
  channelId: string;
  guildId?: string;
}

@RegisterSchema()
export class ChannelTarget implements ChannelTargetLike {
  @SchemaProperty({ description: 'Channel ID. ', required: true })
  channelId: string;
  @SchemaProperty({ description: 'Guild ID. ' })
  guildId?: string;

  toDesc(): string | [string, string] {
    return this.guildId ? [this.channelId, this.guildId] : this.channelId;
  }

  toFilter(): Filter {
    return (s) => {
      if (this.channelId !== s.channelId) {
        return false;
      }
      if (this.guildId && this.guildId !== s.guildId) {
        return false;
      }
      return true;
    };
  }
}

export interface SendTargetLike {
  bot?: string;
  users?: string[];
  channels?: ChannelTargetLike[];
}

@RegisterSchema()
export class SendTarget {
  constructor(_: SendTargetLike) {}

  @SchemaProperty({
    description:
      'Bot identifier. eg. onebot:123456789. Will use first bot if not specified. s',
  })
  bot?: string;

  @SchemaProperty({
    type: String,
    description: 'Private chat send targets. ',
    default: [],
  })
  users?: string[];

  @SchemaProperty({
    type: ChannelTarget,
    description: 'Channel send targets. ',
    default: [],
  })
  channels?: ChannelTarget[];

  getBot(bots: Bot[]) {
    return bots.find((bot) =>
      this.bot ? bot.sid === this.bot : !bot['parentBot'],
    );
  }

  private isOneBotBot(bot?: Bot) {
    return (
      bot &&
      (bot.platform === 'onebot' || bot['parentBot']?.platform === 'onebot')
    );
  }

  getFilter(): Filter {
    let filters: Filter[] = [];
    if (this.users?.length) {
      const userSet = new Set(this.users);
      filters.push((s) => !s.guildId && userSet.has(s.userId));
    }
    if (this.channels?.length) {
      filters = filters.concat(this.channels.map((c) => c.toFilter()));
    }
    return (s) => {
      if (this.bot && `${s.platform}:${s.selfId}` !== this.bot) {
        return false;
      }
      return filters.some((f) => f(s));
    };
  }

  getContext(ctx: Context) {
    return ctx.intersect(this.getFilter());
  }

  async send(bots: Bot[], content: string) {
    const bot = this.getBot(bots);
    if (!bot) {
      throw new Error(`bot ${this.bot} not found`);
    }
    let privateContent = content;
    let channelContent = content;
    if (this.isOneBotBot(bot)) {
      content = content.replace(/,url=base64/g, ',file=base64');
      privateContent = content.replace(/,url=http/g, ',file=http'); // private should be file
      channelContent = content.replace(/,file=http/g, ',url=http'); // channel should be url
    }
    return _.flatten(
      await Promise.all([
        ...this.users.map((userId) =>
          bot.sendPrivateMessage
            ? bot.sendPrivateMessage(userId, privateContent)
            : bot.sendMessage(`@${userId}`, privateContent),
        ),
        ...(this.channels?.length
          ? [
              bot.broadcast(
                this.channels.map((c) => c.toDesc()),
                channelContent,
              ),
            ]
          : []),
      ]),
    );
  }
}
