import { Adapter } from 'koishi';
import _ from 'lodash';
import { RegisterSchema, SchemaProperty } from 'schemastery-gen';

@RegisterSchema()
export class ChannelTarget {
  @SchemaProperty({ description: 'Channel ID. ', required: true })
  channelId: string;
  @SchemaProperty({ description: 'Guild ID .' })
  guildId: string;

  toDesc(): string | [string, string] {
    return this.guildId ? [this.channelId, this.guildId] : this.channelId;
  }
}

@RegisterSchema()
export class SendTarget {
  @SchemaProperty({
    required: true,
    description: 'Bot identifier. eg. onebot:123456789. ',
  })
  bot: string;

  @SchemaProperty({
    type: String,
    description: 'Private chat send targets. ',
    default: [],
  })
  users: string[];

  @SchemaProperty({
    type: ChannelTarget,
    description: 'Channel send targets. ',
    default: [],
  })
  channels: ChannelTarget[];

  async send(bots: Adapter.BotList, content: string) {
    const bot = bots.get(this.bot);
    if (!bot) {
      throw new Error(`bot ${this.bot} not found`);
    }
    return _.flatten(
      await Promise.all([
        ...this.users.map((userId) =>
          bot.sendPrivateMessage
            ? bot.sendPrivateMessage(userId, content)
            : bot.sendMessage(`@${userId}`, content),
        ),
        ...(this.channels?.length
          ? [
              bot.broadcast(
                this.channels.map((c) => c.toDesc()),
                content,
              ),
            ]
          : []),
      ]),
    );
  }
}
