import { Bot, Context, Filter, segment } from 'koishi';
import _ from 'lodash';
import { RegisterSchema, SchemaProperty } from 'schemastery-gen';

export interface ChannelTargetLike {
  channelId: string;
  guildId?: string;
}

@RegisterSchema()
export class ChannelTarget implements ChannelTargetLike {
  @SchemaProperty({ description: 'Channel ID. ', required: true })
  channelId: string;
  @SchemaProperty({ description: 'Guild ID. ' })
  guildId?: string;

  toDesc(): string | [string, string] {
    return this.guildId ? [this.channelId, this.guildId] : this.channelId;
  }

  toFilter(): Filter {
    return (s) => {
      if (this.channelId !== s.channelId) {
        return false;
      }
      if (this.guildId && this.guildId !== s.guildId) {
        return false;
      }
      return true;
    };
  }
}

export interface SendTargetLike {
  bot?: string;
  users?: string[];
  channels?: ChannelTargetLike[];
}

@RegisterSchema()
export class SendTarget {
  constructor(_: SendTargetLike) {}

  @SchemaProperty({
    description:
      'Bot identifier. eg. onebot:123456789. Will use first bot if not specified. s',
  })
  bot?: string;

  @SchemaProperty({
    type: String,
    description: 'Private chat send targets. ',
    default: [],
  })
  users?: string[];

  @SchemaProperty({
    type: ChannelTarget,
    description: 'Channel send targets. ',
    default: [],
  })
  channels?: ChannelTarget[];

  getBot(bots: Bot[]) {
    return bots.find((bot) =>
      this.bot ? bot.sid === this.bot : !bot['parentBot'],
    );
  }

  private isOneBotBot(bot?: Bot) {
    return (
      bot &&
      (bot.platform === 'onebot' || bot['parentBot']?.platform === 'onebot')
    );
  }

  getFilter(): Filter {
    let filters: Filter[] = [];
    if (this.users?.length) {
      const userSet = new Set(this.users);
      filters.push((s) => !s.guildId && userSet.has(s.userId));
    }
    if (this.channels?.length) {
      filters = filters.concat(this.channels.map((c) => c.toFilter()));
    }
    return (s) => {
      if (this.bot && `${s.platform}:${s.selfId}` !== this.bot) {
        return false;
      }
      return filters.some((f) => f(s));
    };
  }

  getContext(ctx: Context) {
    return ctx.intersect(this.getFilter());
  }

  private runForEachSegemnt(segments: segment[], action: (s: segment) => void) {
    for (const s of segments) {
      action(s);
      if (s.children?.length) {
        this.runForEachSegemnt(s.children, action);
      }
    }
  }

  replaceContent(content: string, fromKey: string, toKey: string) {
    const segments = segment.parse(content);
    this.runForEachSegemnt(segments, (s) => {
      const { attrs } = s;
      if (!attrs || s.type !== 'image') {
        return;
      }
      if (attrs.url?.startsWith('base64')) {
        const { url } = attrs;
        attrs.file = url;
        delete attrs.url;
      }
      if (attrs[fromKey]?.startsWith('http')) {
        const { [fromKey]: url } = attrs;
        attrs[toKey] = url;
        delete attrs[fromKey];
      }
    });
    return segments.map((s) => s.toString()).join('');
  }

  async send(bots: Bot[], content: string) {
    const bot = this.getBot(bots);
    if (!bot) {
      throw new Error(`bot ${this.bot} not found`);
    }
    return _.flatten(
      await Promise.all([
        ...this.users.map((userId) =>
          bot.sendPrivateMessage
            ? bot.sendPrivateMessage(userId, content)
            : bot.sendMessage(`@${userId}`, content),
        ),
        ...(this.channels?.length
          ? [
              bot.broadcast(
                this.channels.map((c) => c.toDesc()),
                content,
              ),
            ]
          : []),
      ]),
    );
  }
}
