// import 'source-map-support/register';
import { Context, Random } from 'koishi';
import { SendTarget } from 'koishi-target-def';
import { RegisterSchema, SchemaProperty } from 'koishi-thirdeye';
import { scheduleJob } from 'node-schedule';

@RegisterSchema()
export class RandomInterval {
  @SchemaProperty({ step: 1, default: 0 })
  max: number;

  @SchemaProperty({ step: 1, default: 0 })
  min: number;

  getInterval() {
    return Random.int(this.min, this.max + 1);
  }

  isAvailable() {
    return !!this.max && !!this.min;
  }

  processRandom(ctx: Context, callback: () => void) {
    ctx.setTimeout(() => {
      callback();
      this.processRandom(ctx, callback);
    }, this.getInterval());
  }
}

export class SchedulePluginConfig {
  @SchemaProperty({ description: 'cron 时间。' })
  cron: string;

  @SchemaProperty({ description: '执行的间隔 ms。', step: 1 })
  interval: number;

  @SchemaProperty({ description: '随机间隔。' })
  randomInterval: RandomInterval;

  @SchemaProperty({
    default: false,
    description: '插件启动时是否立即运行一遍。',
  })
  immediate: boolean;

  @SchemaProperty({
    type: SendTarget,
    description: '图片发送目标。',
  })
  targets: SendTarget[];

  initializeTasks(ctx: Context, callback: () => void) {
    if (this.immediate) {
      callback();
    }
    if (this.cron) {
      const job = scheduleJob(this.cron, callback);
      ctx.on('dispose', () => {
        job.cancel();
      });
    }
    if (this.interval) {
      ctx.setInterval(callback, this.interval);
    }
    if (this.randomInterval?.isAvailable()) {
      this.randomInterval.processRandom(ctx, callback);
    }
  }
}
