import 'source-map-support/register';
import { Credentials } from '@aws-sdk/types';
import {
  DefineSchema,
  SchemaConf,
  schemaFromClass,
  UseSchema,
} from 'koishi-utils-schemagen';
import path from 'path';
import { Context, Schema } from 'koishi';
import {
  GetObjectCommand,
  ListObjectsCommand,
  S3Client,
  S3ClientConfig,
} from '@aws-sdk/client-s3';
import { SRVProRoomInfo } from './def/srvpro';
import { Selection } from './def/selection';

const credentialsSchema: Schema<Credentials> = Schema.object(
  {
    accessKeyId: Schema.string().required(),
    secretAccessKey: Schema.string().required(),
  },
  true,
);

export const s3Schema: Schema<S3ClientConfig> = Schema.object(
  {
    region: Schema.string().default('none'),
    endpoint: Schema.string(),
    credentials: credentialsSchema,
  },
  true,
);

@SchemaConf({ desc: '发送卡组的置' })
export class DeckFetchConfig {
  @DefineSchema({ desc: 'S3 配置', schema: s3Schema })
  s3: S3ClientConfig;

  @DefineSchema({ desc: '请求的 Host 字段' })
  host: string;

  @DefineSchema({ desc: '存储桶' })
  bucket: string;

  @DefineSchema({ desc: '卡组路径前缀', default: '' })
  prefix: string;

  @DefineSchema({ desc: 'URL 有效期', default: 600 })
  urlAge: number;

  S3Client(): S3Client {
    const client = new S3Client({
      region: 'none',
      forcePathStyle: true,
      ...this.s3,
    });
    return client;
  }

  listCommand() {
    return new ListObjectsCommand({
      Bucket: this.bucket,
      Prefix: this.prefix,
    });
  }

  getObjectCommand(key: string) {
    return new GetObjectCommand({
      Bucket: this.bucket,
      Key: key,
    });
  }
}

@SchemaConf({ desc: '比赛过程的设置' })
export class TournamentConfig {
  @DefineSchema({
    desc: '服务器 API 连接后台，不带任何路径和尾随斜杠',
  })
  endpoint: string;

  @DefineSchema({ desc: '服务器后台用户名' })
  username: string;

  @DefineSchema({ desc: '服务器后台密码' })
  password: string;

  @DefineSchema({
    desc: 'Challonge API Key',
  })
  challongeKey: string;

  @DefineSchema({
    desc: 'Challonge 比赛 ID',
  })
  challongeTournamentId: string;

  @DefineSchema({
    desc: 'Challonge 缓存时间',
    default: 30000,
  })
  challongeCacheTTL: number;

  @DefineSchema({
    desc: 'Challonge URL 前缀',
    default: 'https://api.challonge.com/v1/tournaments/',
  })
  challongeUrlPrefix: string;

  getChallongeUrl() {
    return `${this.challongeUrlPrefix}${this.challongeTournamentId}`;
  }

  async fetchRooms(ctx: Context) {
    return ctx.http.get<Partial<SRVProRoomInfo>>(
      `${this.endpoint}/api/getrooms`,
      {
        username: this.username,
        pass: this.password,
      },
    );
  }

  async kickRoom(ctx: Context, search: string) {
    return ctx.http.get(`${this.endpoint}/api/message`, {
      username: this.username,
      pass: this.password,
      kick: search,
    });
  }
}

export class YGOTournamentPluginConfig {
  @DefineSchema({
    type: 'object',
    allowUnknown: true,
    desc: '裁判接口作用域',
    required: true,
  })
  judgeSelection: Selection;

  @DefineSchema()
  tournament: TournamentConfig;

  @DefineSchema()
  deckFetch: DeckFetchConfig;

  isTournamentEnabled() {
    return !!this.tournament;
  }

  isDeckFetchEnabled() {
    return !!this.deckFetch;
  }
}

export interface YGOTournamentPluginConfigLike {
  judgeSelection: Selection;
  tournament?: Partial<TournamentConfig>;
  deckFetch?: Partial<DeckFetchConfig>;
}
