export * from './config';
import { YGOTournamentPluginConfig } from './config';
import { plainToInstance } from 'class-transformer';
import { MatchWrapper, TournamentWrapper } from './def/challonge';
import { S3Client } from '@aws-sdk/client-s3';
import { getSignedUrl } from './presign';
import { SRVProRoomInfo } from './def/srvpro';
import moment from 'moment';
import {
  DefinePlugin,
  Inject,
  selectContext,
  StarterPlugin,
} from 'koishi-thirdeye';
import AragamiPlugin, { CacheKey } from 'koishi-plugin-cache-aragami';

class LateDeclarationTime {
  fromTime: number;
  toTime: number;
  @CacheKey()
  key() {
    return 'current';
  }
}

class ChallongeData extends TournamentWrapper {
  @CacheKey()
  challongeId: string;
}

@DefinePlugin()
export default class YGOTournamentPlugin extends StarterPlugin(
  YGOTournamentPluginConfig,
) {
  @Inject(true)
  private aragami: AragamiPlugin;

  private s3: S3Client;
  private async getChallongeCache() {
    return this.aragami.get(
      ChallongeData,
      this.config.tournament.challongeTournamentId,
    );
  }

  private async deleteChallongeCache() {
    return this.aragami.get(
      ChallongeData,
      this.config.tournament.challongeTournamentId,
    );
  }

  private async setChallongeCache(data: TournamentWrapper) {
    return this.aragami.set(ChallongeData, {
      ...data,
      challongeId: this.config.tournament.challongeTournamentId,
    });
  }

  private async fetchChallongeData() {
    const cached = await this.getChallongeCache();
    if (cached) {
      return cached;
    }
    const data = await this.ctx.http.get(
      `${this.config.tournament.getChallongeUrl()}.json`,
      {
        params: {
          api_key: this.config.tournament.challongeKey,
          include_participants: 1,
          include_matches: 1,
        },
      },
    );
    return this.setChallongeCache(data);
  }

  private async onUserQuit(userId: string) {
    const tournament = await this.fetchChallongeData();
    const participant = tournament.getParticipantFromId(userId);
    if (!participant) {
      return '未找到您的参赛信息。';
    }
    // const participantName = participant.getNameMatching().userName;
    this.ctx
      .logger('challonge')
      .info(`Player ${participant.getDisplayName()} requested for quit.`);
    const currentMatch = tournament.getCurrentMatchFromId(
      participant.participant.id,
    );
    if (currentMatch) {
      const scores_csv =
        participant.participant.id === currentMatch.match.player2_id
          ? '2--1'
          : '-1-2';
      const winner_id = currentMatch
        .getPlayerIds()
        .find((id) => id !== participant.participant.id);
      this.ctx
        .logger('challonge')
        .info(
          `Cleaning remaining match for player ${participant.getDisplayName()}.`,
        );
      await this.ctx.http.put(
        `${this.config.tournament.getChallongeUrl()}/matches/${
          currentMatch.match.id
        }.json`,
        {
          api_key: this.config.tournament.challongeKey,
          match: { scores_csv, winner_id },
        },
      );
      this.ctx
        .logger('challonge')
        .info(
          `Cleaned remaining match for player ${participant.participant.name}.`,
        );
    }
    await this.ctx.http.delete(
      `${this.config.tournament.getChallongeUrl()}/participants/${
        participant.participant.id
      }.json`,
      {
        params: {
          api_key: this.config.tournament.challongeKey,
        },
      },
    );
    this.ctx
      .logger('challonge')
      .info(`Player ${participant.getDisplayName()} quited.`);
    await this.deleteChallongeCache();
    return '退赛成功。';
  }

  private async onUserGetCurrentMatch(userId: string) {
    const tournament = await this.fetchChallongeData();
    const participant = tournament.getParticipantFromId(userId);
    if (!participant) {
      return '未找到您的参赛信息。';
    }
    const currentMatch = tournament.getCurrentMatchFromId(
      participant.participant.id,
      true,
    );
    if (!currentMatch) {
      return '您没有正在进行的比赛。';
    }
    return `您当前的对局信息:\n${tournament.displayMatch(currentMatch)}`;
  }

  private async fetchSRVProRoomlist() {
    const rawRoomlist = await this.config.tournament.fetchRooms(this.ctx);
    return plainToInstance(SRVProRoomInfo, rawRoomlist);
  }

  private async onUserDeclareLate(userId: string) {
    const timeUnix = await this.aragami.get(LateDeclarationTime, 'current');
    // this.ctx.logger('test').warn(JSON.stringify(timeUnix));
    if (
      !timeUnix ||
      !moment().isBetween(
        moment.unix(timeUnix.fromTime),
        moment.unix(timeUnix.toTime),
      )
    ) {
      return '现在不是允许迟到杀的时间。';
    }
    const tournament = await this.fetchChallongeData();
    const participant = tournament.getParticipantFromId(userId);
    if (!participant) {
      return '未找到您的参赛信息。';
    }
    const participantName = participant.getNameMatching().userName;
    const currentMatch = tournament.getCurrentMatchFromId(
      participant.participant.id,
    );
    if (!currentMatch) {
      return '你没有当前轮次的比赛。';
    }
    if (!currentMatch.isClean()) {
      return '你涉及的比赛似乎已经有结果了，请联系裁判处理。';
    }
    const roomlist = await this.fetchSRVProRoomlist();
    const roomsWithPlayer = roomlist.searchForUser(participantName);
    if (!roomsWithPlayer.length) {
      return '请先进入房间并打勾确认。';
    }
    if (
      roomsWithPlayer.some((r) => r.istart !== 'wait' || r.users.length > 1)
    ) {
      return '对局已经开始，不能迟到杀。';
    }
    const scores_csv =
      participant.participant.id === currentMatch.match.player1_id
        ? '2--1'
        : '-1-2';
    const rivalId = currentMatch
      .getPlayerIds()
      .find((id) => id !== participant.participant.id);
    const rival = tournament.tournament.participants.find(
      (p) => p.participant.id === rivalId,
    );
    this.ctx
      .logger('challonge')
      .info(
        `Player ${participant.getDisplayName()} requested for a late declaration against ${rival?.getDisplayName()}.`,
      );
    const result = await this.ctx.http.put(
      `${this.config.tournament.getChallongeUrl()}/matches/${
        currentMatch.match.id
      }.json`,
      {
        api_key: this.config.tournament.challongeKey,
        match: { scores_csv, winner_id: participant.participant.id },
      },
    );

    this.ctx
      .logger('challonge')
      .info(
        `Player ${participant.getDisplayName()} declared late against ${rival?.getDisplayName()}.`,
      );

    await Promise.all([
      this.deleteChallongeCache(),
      ...roomsWithPlayer.map((r) =>
        this.config.tournament.kickRoom(this.ctx, r.roomid),
      ),
    ]);
    return '迟到杀成功。';
  }

  private async getZombieMatches() {
    const tournament = await this.fetchChallongeData();
    const matches = tournament.getZombieMatches();
    if (!matches.length) {
      return { tournament, matches };
    }
    const roomlist = await this.fetchSRVProRoomlist();
    const zombieMatches = matches.filter(
      (m) =>
        !roomlist.rooms.some(
          (r) =>
            r.istart !== 'wait' && r.roomname.endsWith(m.match.id.toString()),
        ),
    );
    return { tournament, matches: zombieMatches };
  }

  private async cleanZombieMatch(match: MatchWrapper) {
    await this.ctx.http.put(
      `${this.config.tournament.getChallongeUrl()}/matches/${
        match.match.id
      }.json`,
      {
        api_key: this.config.tournament.challongeKey,
        match: { scores_csv: '0-0', winner_id: 'tie' },
      },
    );
  }

  private initializeDeckFetch() {
    if (!this.config.isDeckFetchEnabled()) {
      return;
    }
    this.s3 = this.config.deckFetch.S3Client();
    this.ctx
      .command('tournament/deck', '获取自己的卡组')
      .action(async ({ session }) => {
        const userId = session.userId;
        try {
          const { Contents } = await this.s3.send(
            this.config.deckFetch.listCommand(),
          );
          const deck = Contents.find(
            (c) => c.Key.endsWith('.ydk') && c.Key.includes(userId),
          );
          if (!deck) {
            return '未找到您的卡组。';
          }
          const url = await getSignedUrl(
            this.s3,
            this.config.deckFetch.getObjectCommand(deck.Key),
            {
              expiresIn: this.config.deckFetch.urlAge,
              extraHeaders: this.config.deckFetch.host && {
                host: this.config.deckFetch.host,
              },
            },
          );
          return `获取卡组成功。您可以在下列地址下载您的卡组:\n${url}`;
        } catch (e) {
          this.ctx
            .logger('deckfetch')
            .error(`Failed to fetch deck of ${userId}: ${e.toString()}`);
          return '获取卡组出现了问题。请与技术人员联系。';
        }
      });
  }

  initializeTournament() {
    if (!this.config.isTournamentEnabled()) {
      return;
    }
    this.ctx
      .command('tournament/currentmatch', '获取当前对局')
      .shortcut('获取当前对局')
      .usage('获取自己的当前对局和比分。')
      .action(async ({ session }) => {
        try {
          return await this.onUserGetCurrentMatch(session.userId);
        } catch (e) {
          this.ctx
            .logger('challonge')
            .error(
              `Failed to fetch current match of ${
                session.userId
              }: ${e.toString()}`,
            );
          return '获取比赛出现了一些问题，请与技术人员联系。';
        }
      });
    this.ctx
      .command('tournament/late', '迟到杀')
      .shortcut('迟到杀')
      .usage('迟到杀需要选手在房间内，且只有1人才可以迟到杀。')
      .action(async ({ session }) => {
        try {
          return await this.onUserDeclareLate(session.userId);
        } catch (e) {
          this.ctx
            .logger('challonge')
            .error(`Failed to quit user ${session.userId}: ${e.toString()}`);
          return '迟到杀出现了一些问题，请与技术人员联系。';
        }
      });
    this.ctx
      .command('tournament/quit', '退赛')
      .shortcut('退赛')
      .usage('退赛需要慎重，会取消后续的比赛资格。')
      .action(async ({ session }) => {
        await session.send('确定要退赛吗？输入 yes 以确认。');
        const reply = await session.prompt();
        if (reply !== 'yes') {
          return '退赛被取消，接下来要继续努力哦。';
        }
        try {
          return await this.onUserQuit(session.userId);
        } catch (e) {
          this.ctx
            .logger('challonge')
            .error(`Failed to quit user ${session.userId}: ${e.toString()}`);
          return '退赛出现了一些问题，请与技术人员联系。';
        }
      });
    const judgeCommand = selectContext(this.ctx, this.config.judgeSelection)
      .command('tournament/judge', '裁判操作')
      .usage('需要有裁判权限才能执行这些操作。');
    judgeCommand
      .subcommand('.enablelate', '开启迟到杀')
      .usage('允许选手使用指令来进行迟到杀。')
      .option('delay', '-d <time:posint>  在指定分钟后开启迟到杀。', {
        fallback: 0,
      })
      .option('duration', '-t <time:posint>  迟到杀允许的分钟数。', {
        fallback: 30,
      })
      .action(async ({ session, options }) => {
        const fromTime = moment().add(options.delay, 'minutes');
        const toTime = fromTime.clone().add(options.duration, 'minutes');
        await this.aragami.set(
          LateDeclarationTime,
          {
            fromTime: fromTime.unix(),
            toTime: toTime.unix(),
          },
          { ttl: (options.delay + options.duration) * 60000 },
        );
        return `设置成功。将允许在 ${fromTime.format(
          'HH:mm:ss',
        )} 至 ${toTime.format('HH:mm:ss')} 期间允许迟到杀。`;
      });
    judgeCommand
      .subcommand('.disablelate', '关闭迟到杀')
      .usage('不再允许选手进行迟到杀操作。')
      .action(async () => {
        await this.aragami.del(LateDeclarationTime, 'current');
        return '设置成功。选手不再允许进行迟到杀操作。';
      });
    judgeCommand
      .subcommand('.refresh', '清理 Challonge 缓存')
      .action(async () => {
        await this.aragami.del(
          ChallongeData,
          this.config.tournament.challongeTournamentId,
        );
        return '清理缓存成功。';
      });
    judgeCommand
      .subcommand('.cleanzombie', '清理僵尸对局')
      .usage('会清理所有没有开始的对局，统一设置为 0-0 平。')
      .action(async ({ session }) => {
        try {
          const { tournament, matches } = await this.getZombieMatches();
          if (!matches.length) {
            return '未找到僵尸对局。';
          }
          await session.send(
            `找到了 ${matches.length} 个僵尸对局:\n${matches
              .map((m) => tournament.displayMatch(m))
              .join('\n')}\n\n输入 yes 进行清理。`,
          );
          const reply = await session.prompt();
          if (reply !== 'yes') {
            return '清理被取消。';
          }
          const logger = this.ctx.logger('challonge');
          logger.info(`Cleaning ${matches.length} zombie matches.`);
          await Promise.all(matches.map((m) => this.cleanZombieMatch(m)));
          logger.info(`Cleaned ${matches.length} zombie matches.`);
          return `清理成功，清理了 ${matches.length} 个僵尸对局。`;
        } catch (e) {
          this.ctx
            .logger('challonge')
            .error(`Failed to clean zombie matches: ${e.toString()}`);
        }
      });
  }

  onApply() {
    this.ctx.command('tournament', 'YGOPro 比赛相关命令');
    this.initializeTournament();
    this.initializeDeckFetch();
  }
}
