// import 'source-map-support/register';
import { Context, Schema } from 'koishi';
import { Selection } from 'koishi-thirdeye';
import {
  GetObjectCommand,
  ListObjectsCommand,
  S3Client,
  S3ClientConfig,
} from '@aws-sdk/client-s3';
import { SRVProRoomInfo } from './def/srvpro';
import { DefineSchema, RegisterSchema } from 'schemastery-gen';

const credentialsSchema = Schema.object({
  accessKeyId: Schema.string().required().role('secret'),
  secretAccessKey: Schema.string().required().role('secret'),
});

export const s3Schema = Schema.object({
  region: Schema.string().default('none'),
  endpoint: Schema.string().role('url'),
  credentials: credentialsSchema,
});

@RegisterSchema({ description: '发送卡组的置' })
export class DeckFetchConfig {
  constructor(config: Partial<DeckFetchConfig>) {}
  @DefineSchema({ description: 'S3 配置', type: s3Schema })
  s3: S3ClientConfig;

  @DefineSchema({ description: '请求的 Host 字段' })
  host: string;

  @DefineSchema({ description: '存储桶' })
  bucket: string;

  @DefineSchema({ description: '卡组路径前缀', default: '' })
  prefix: string;

  @DefineSchema({ description: 'URL 有效期', default: 600 })
  urlAge: number;

  S3Client(): S3Client {
    const client = new S3Client({
      region: 'none',
      forcePathStyle: true,
      ...this.s3,
    });
    return client;
  }

  listCommand() {
    return new ListObjectsCommand({
      Bucket: this.bucket,
      Prefix: this.prefix,
    });
  }

  getObjectCommand(key: string) {
    return new GetObjectCommand({
      Bucket: this.bucket,
      Key: key,
    });
  }
}

@RegisterSchema({ description: '比赛过程的设置' })
export class TournamentConfig {
  constructor(config: Partial<TournamentConfig>) {}
  @DefineSchema({
    description: '服务器 API 连接后台，不带任何路径和尾随斜杠',
    role: 'url',
  })
  endpoint: string;

  @DefineSchema({ description: '服务器后台用户名' })
  username: string;

  @DefineSchema({ description: '服务器后台密码', role: 'secret' })
  password: string;

  @DefineSchema({
    description: 'Challonge API Key',
  })
  challongeKey: string;

  @DefineSchema({
    description: 'Challonge 比赛 ID',
  })
  challongeTournamentId: string;

  @DefineSchema({
    description: 'Challonge 缓存时间',
    default: 30000,
  })
  challongeCacheTTL: number;

  @DefineSchema({
    description: 'Challonge URL 前缀',
    default: 'https://api.challonge.com/v1/tournaments/',
  })
  challongeUrlPrefix: string;

  getChallongeUrl() {
    return `${this.challongeUrlPrefix}${this.challongeTournamentId}`;
  }

  async fetchRooms(ctx: Context) {
    return ctx.http.get<Partial<SRVProRoomInfo>>(
      `${this.endpoint}/api/getrooms`,
      {
        params: {
          username: this.username,
          pass: this.password,
        },
      },
    );
  }

  async kickRoom(ctx: Context, search: string) {
    return ctx.http.get(`${this.endpoint}/api/message`, {
      params: {
        username: this.username,
        pass: this.password,
        kick: search,
      },
    });
  }
}

@RegisterSchema()
export class YGOTournamentPluginConfig {
  constructor(config: Partial<YGOTournamentPluginConfig>) {}
  @DefineSchema({
    type: 'object',
    description: '裁判接口作用域',
    required: true,
  })
  judgeSelection: Selection;

  @DefineSchema()
  tournament: TournamentConfig;

  @DefineSchema()
  deckFetch: DeckFetchConfig;

  isTournamentEnabled() {
    return !!this.tournament;
  }

  isDeckFetchEnabled() {
    return !!this.deckFetch;
  }
}

export interface YGOTournamentPluginConfigLike {
  judgeSelection: Selection;
  tournament?: Partial<TournamentConfig>;
  deckFetch?: Partial<DeckFetchConfig>;
}
