// import 'source-map-support/register';
import { YGOCardConfig } from './config';
import SQL from 'sql.js';
import { promises as fs } from 'fs';
import _ from 'lodash';
import { YGOProCard } from './YGOProCard';
import { plainToInstance } from 'class-transformer';
import path from 'path';
import {
  StarterPlugin,
  DefinePlugin,
  LifecycleEvents,
  InjectLogger,
  UseCommand,
  PutOption,
  PutArg,
  PutSession,
  CommandLocale,
  Inject,
  Provide,
} from 'koishi-thirdeye';
import { Logger, Random, Session } from 'koishi';
import initSqlJs from 'sql.js';
import * as localeZh from './locales/zh';
import * as localeEn from './locales/en';
import PuppeteerPlugin from 'koishi-plugin-puppeteer';

export * from './config';

declare module 'koishi' {
  interface Context {
    ygocard: YGOCardPlugin;
  }
}

const commonFilter =
  'datas.type & 0x4000 = 0 and (datas.alias = 0 or datas.id - datas.alias > 10)';

@Provide('ygocard')
@DefinePlugin({ name: 'ygocard', schema: YGOCardConfig })
export default class YGOCardPlugin
  extends StarterPlugin(YGOCardConfig)
  implements LifecycleEvents
{
  @InjectLogger()
  private logger: Logger;

  @Inject()
  private puppeteer: PuppeteerPlugin;

  private dbs: SQL.Database[] = [];
  private querySQL(db: SQL.Database, sql: string, params: any = {}) {
    const statement = db.prepare(sql);
    statement.bind(params);
    const results: any[] = [];
    while (statement.step()) {
      results.push(statement.getAsObject());
    }
    statement.free();
    return results;
  }
  private queryInDB(value: string, db: SQL.Database) {
    const possibleValueNumber = parseInt(value) || 0;
    const results = this.querySQL(
      db,
      `select datas.*,texts.name,texts.desc from datas,texts where datas.id = texts.id and ${commonFilter} and (datas.id = $num or texts.name like ('%' || $query || '%') or texts.desc like ('%' || $query || '%'))`,
      {
        $num: possibleValueNumber,
        $query: value,
      },
    );
    return results.map((obj) => plainToInstance(YGOProCard, obj));
  }
  private randomCardInDB(db: SQL.Database) {
    const allCardIds = this.querySQL(
      db,
      `select id from datas where ${commonFilter}`,
    );
    if (!allCardIds.length) return;
    const id: number = Random.pick(allCardIds).id;
    const [result] = this.queryInDB(id.toString(), db);
    return result;
  }
  randomCard() {
    const db = Random.pick(this.dbs);
    if (!db) return;
    return this.randomCardInDB(db);
  }
  queryInAllDBs(value: string, matchCount?: number) {
    return _.sortBy(
      _.uniqBy(
        _.flatten(this.dbs.map((db) => this.queryInDB(value, db))),
        (c) => c.id,
      ),
      (c) => c.getDistanceFrom(value),
    ).slice(0, matchCount || this.config.matchCount);
  }
  private async loadDatabases() {
    const allDatabaseFiles = _.flatten(
      await Promise.all(
        this.config.databasePaths.map(async (p) =>
          (await fs.readdir(p))
            .filter((f) => f.endsWith('.cdb'))
            .map((f) => path.join(p, f)),
        ),
      ),
    );
    const buffers = await Promise.all(
      allDatabaseFiles.map((f) => fs.readFile(f)),
    );
    this.dbs = buffers.map((b) => new this.SQL.Database(b));
    this.logger.info(`Loaded cards from ${allDatabaseFiles.length} databases.`);
  }

  private SQL: initSqlJs.SqlJsStatic;

  async onConnect() {
    this.SQL = await initSqlJs();
    return this.loadDatabases();
  }

  private disposeDatabase() {
    this.dbs.forEach((db) => db.close());
    this.dbs = [];
  }

  onDisconnect() {
    this.disposeDatabase();
  }

  async reloadDatabase() {
    this.disposeDatabase();
    await this.loadDatabases();
  }

  async renderCard(card: YGOProCard) {
    if (!this.config.usePuppeteer || !this.puppeteer) {
      return card.getDisplayString(this.config);
    }
    const img = card.getPic(this.config);
    const { name, id, desc } = card;
    const meta = card.getMetaText(this.config);
    return (
      <html>
        <div>
          <style>{`
        .img  {
          grid-area: img;
          width: 150px;
          border-radius: 5px;
          border: solid 1px gray;
          padding: 5px;
        }
        .name { grid-area: name; }
        .hr   {
          grid-area: hr;
          width: 100%;
        }
        .id   { grid-area: id; }
        .meta { grid-area: meta; }
        .desc {
          grid-area: desc;
          border-radius: 5px;
          border: solid 1px gray;
          padding: 10px;
        }

        .main {
          display: grid;
          grid-template-areas:
            'img  name name name'
            'img  name name name'
            'img  hr   hr   hr  '
            'img  meta meta meta'
            'img  none none none'
            'desc desc desc desc';
          gap: 10px;
          width: 400px;
          padding: 20px;
        }
      `}</style>
          <div class="main">
            <image class="img" src={img} />
            <div class="name">
              <h3>{name}</h3>
              <div>{id}</div>
            </div>
            <hr class="hr" />
            <div class="meta">
              {...meta.split('\n').map((line) => <div>{line}</div>)}
            </div>
            <div class="desc">
              {...desc.split('\n').map((line) => <div>{line}</div>)}
            </div>
          </div>
        </div>
      </html>
    );
  }

  @UseCommand('{{commandName}} [query]')
  @CommandLocale('zh', localeZh.cardCommand)
  @CommandLocale('en', localeEn.cardCommand)
  private async onCardCommand(
    @PutArg(0) query: string,
    @PutOption('random', '--random') random: boolean,
    @PutOption('count', '-c <count:posint>') count: number,
    @PutSession() session: Session,
  ) {
    if (!query && !random) {
      await session.send(session.text('.prompt-input'));
      query = await session.prompt();
    }
    if (!query && !random) return;
    if (!count) {
      count = this.config.matchCount;
    }
    const cards = random
      ? [this.randomCard()]
      : this.queryInAllDBs(query, count);
    if (!cards.length) {
      return session.text('.not-found');
    }
    if (cards.length === 1) {
      return this.renderCard(cards[0]);
    }
    const itemLines = cards.map((c, i) => `${i + 1}. ${c.getIdAndName()}`);
    const borderLength = Math.max(...itemLines.map((l) => l.length)) + 1;
    await session.send(
      session.text('.select-prefix-part1') +
        query +
        session.text('.select-prefix-part2') +
        '\n' +
        _.repeat('-', borderLength) +
        '\n' +
        itemLines.join('\n') +
        '\n' +
        _.repeat('-', borderLength) +
        '\n' +
        session.text('.select-suffix'),
    );
    const reply = await session.prompt();
    if (!reply) {
      return;
    }
    const replyNumber = parseInt(reply);
    const card =
      cards[replyNumber - 1] ||
      cards.find((c) => c.id === replyNumber) ||
      cards.find((c) => c.name === reply) ||
      cards.find((c) => c.name.includes(reply)) ||
      cards.find((c) => c.desc.includes(reply));
    if (!card) {
      return session.text('.not-found');
    }
    return this.renderCard(card);
  }
}
