import { segment } from 'koishi';
import { CardInfoService } from './card-info.service';
import { YGOCardConfig } from './config';
import leven from 'leven';

const cardInfoService = new CardInfoService();

const attrOffset = 1010;
const raceOffset = 1020;
const typeOffset = 1050;

const ygoproConstants = {
  TYPES: {
    TYPE_MONSTER: 1,
    TYPE_SPELL: 2,
    TYPE_TRAP: 4,
    TYPE_NORMAL: 16,
    TYPE_EFFECT: 32,
    TYPE_FUSION: 64,
    TYPE_RITUAL: 128,
    TYPE_TRAPMONSTER: 256,
    TYPE_SPIRIT: 512,
    TYPE_UNION: 1024,
    TYPE_DUAL: 2048,
    TYPE_TUNER: 4096,
    TYPE_SYNCHRO: 8192,
    TYPE_TOKEN: 16384,
    TYPE_QUICKPLAY: 65536,
    TYPE_CONTINUOUS: 131072,
    TYPE_EQUIP: 262144,
    TYPE_FIELD: 524288,
    TYPE_COUNTER: 1048576,
    TYPE_FLIP: 2097152,
    TYPE_TOON: 4194304,
    TYPE_XYZ: 8388608,
    TYPE_PENDULUM: 16777216,
    TYPE_SPSUMMON: 33554432,
    TYPE_LINK: 67108864,
  },
  LINK_MARKERS: {
    LINK_MARKER_BOTTOM_LEFT: 1,
    LINK_MARKER_BOTTOM: 2,
    LINK_MARKER_BOTTOM_RIGHT: 4,
    LINK_MARKER_LEFT: 8,
    LINK_MARKER_RIGHT: 32,
    LINK_MARKER_TOP_LEFT: 64,
    LINK_MARKER_TOP: 128,
    LINK_MARKER_TOP_RIGHT: 256,
  },
};

export interface YGOProCardLike {
  id: number;
  name: string;
  desc: string;
  ot: number;
  alias: number;
  setcode: number;
  type: number;
  atk: number;
  def: number;
  level: number;
  race: number;
  attribute: number;
  category: number;
}

export interface YGOProCardView extends YGOProCardLike {
  cardLScale: number;
  cardRScale: number;
  displayAtk: string;
  displayDef: string;
  displayRace: string;
  displayAttribute: string;
  displayType: string;
  linkMarkers: string;
}

function getStatusValue(s: number) {
  if (s >= 0) {
    return s.toString();
  } else {
    return '?';
  }
}

export class YGOProCard implements YGOProCardLike {
  id: number;
  name: string;
  desc: string;
  ot: number;
  alias: number;
  setcode: number;
  type: number;
  atk: number;
  def: number;
  level: number;
  race: number;
  attribute: number;
  category: number;

  getIdAndName() {
    return `${this.name}[${this.id}]`;
  }

  isType(type: keyof typeof ygoproConstants.TYPES) {
    return !!(this.type & ygoproConstants.TYPES[type]);
  }

  getView(lang: 'cn' | 'en'): Partial<YGOProCardView> {
    const result: Partial<YGOProCardView> = {};
    result.id = this.id;
    result.name = this.name;
    result.desc = this.desc;
    result.ot = this.ot;
    result.alias = this.alias;
    result.setcode = this.setcode;
    result.atk = this.atk;
    result.def = this.def;
    result.type = this.type;
    result.race = this.race;
    result.attribute = this.attribute;
    result.category = this.category;
    if (this.level <= 12) {
      result.level = this.level;
    } else {
      //转化为16位，0x01010004，前2位是左刻度，2-4是右刻度，末2位是等级
      const levelHex = this.level;
      const cardLevel = levelHex & 0xff;
      const cardLScale = (levelHex >> 6) & 0xff;
      const cardRScale = (levelHex >> 4) & 0xff;
      result.level = cardLevel;
      result.cardLScale = cardLScale;
      result.cardRScale = cardRScale;
    }
    result.displayAtk = getStatusValue(this.atk);
    if (!(this.type & ygoproConstants.TYPES.TYPE_LINK)) {
      result.displayDef = getStatusValue(this.def);
    } else {
      // result.name+="[LINK-" + cardLevel + "]";
      // result.name += " " + (result.atk < 0 ? "?" : result.atk) + "/- ";
      result.linkMarkers = '';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_TOP_LEFT)
        result.linkMarkers += '[↖]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_TOP)
        result.linkMarkers += '[↑]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_TOP_RIGHT)
        result.linkMarkers += '[↗]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_LEFT)
        result.linkMarkers += '[←]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_RIGHT)
        result.linkMarkers += '[→]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_BOTTOM_LEFT)
        result.linkMarkers += '[↙]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_BOTTOM)
        result.linkMarkers += '[↓]';
      if (result.def & ygoproConstants.LINK_MARKERS.LINK_MARKER_BOTTOM_RIGHT)
        result.linkMarkers += '[↘]';

      result.displayDef = '-';
    }

    result.displayType = cardInfoService.getStringValueByMysticalNumber(
      lang,
      typeOffset,
      this.type,
    );
    result.displayRace = cardInfoService.getStringValueByMysticalNumber(
      lang,
      raceOffset,
      this.race,
    );
    result.displayAttribute = cardInfoService.getStringValueByMysticalNumber(
      lang,
      attrOffset,
      this.attribute,
    );
    return result;
  }

  getPic(config: YGOCardConfig, view = this.getView(config.getLang())) {
    return segment('image', { url: config.renderUrl(view), cache: true }, []);
  }

  getDisplayString(config: YGOCardConfig, forcePic = false) {
    const view = this.getView(config.getLang());
    const lines: string[] = [];
    if (config.displayPic || forcePic) {
      lines.push(this.getPic(config, view).toString());
    }
    lines.push(`${view.name}[${view.id}]`);
    const isMonster = this.isType('TYPE_MONSTER');
    let typeString = `[${view.displayType}]`;
    if (isMonster) {
      typeString += ` ${view.displayRace}/${view.displayAttribute}`;
    }
    lines.push(typeString);
    if (isMonster) {
      const starString = this.isType('TYPE_XYZ')
        ? '\u2606'
        : this.isType('TYPE_LINK')
        ? 'LINK-'
        : '\u2605';
      let monsterStatus = `[${starString}${view.level}] ${view.displayAtk}/${view.displayDef}`;
      if (this.isType('TYPE_PENDULUM')) {
        monsterStatus += `  ${view.cardLScale}/${view.cardRScale}`;
      }
      if (this.isType('TYPE_LINK')) {
        monsterStatus += `  ${view.linkMarkers}`;
      }
      lines.push(monsterStatus);
    }
    lines.push('');
    lines.push(view.desc);
    return lines.join('\n');
  }

  getDistanceFrom(value: string) {
    if (parseInt(value) === this.id) {
      return 1;
    } else if (this.name.includes(value)) {
      return leven(value, this.name) + 2;
    } else {
      return (leven(value, this.desc) + 2) << 16;
    }
  }
}
