// import 'source-map-support/register';
import { YGOCardConfig } from './config';
import SQL from 'sql.js';
import { promises as fs } from 'fs';
import _ from 'lodash';
import { YGOProCard } from './YGOProCard';
import { plainToInstance } from 'class-transformer';
import path from 'path';
import {
  StarterPlugin,
  DefinePlugin,
  LifecycleEvents,
  InjectLogger,
  UseCommand,
  PutOption,
  PutArg,
  PutSession,
  CommandLocale,
} from 'koishi-thirdeye';
import { Logger, Session } from 'koishi';
import initSqlJs from 'sql.js';
import * as localeZh from './locales/zh';
import * as localeEn from './locales/en';

export * from './config';

@DefinePlugin({ name: 'ygocard', schema: YGOCardConfig })
export default class YGOCardPlugin
  extends StarterPlugin(YGOCardConfig)
  implements LifecycleEvents
{
  @InjectLogger()
  private logger: Logger;

  private dbs: SQL.Database[] = [];
  private queryInDB(value: string, db: SQL.Database) {
    const possibleValueNumber = parseInt(value) || 0;
    const statement = db.prepare(
      "select datas.*,texts.name,texts.desc from datas,texts where datas.id = texts.id and datas.type & 0x4000 = 0 and (datas.alias = 0 or datas.id - datas.alias > 10) and (datas.id = $num or texts.name like ('%' || $query || '%') or texts.desc like ('%' || $query || '%'))",
    );
    statement.bind({
      $num: possibleValueNumber,
      $query: value,
    });
    const results: any[] = [];
    while (statement.step()) {
      results.push(statement.getAsObject());
    }
    statement.free();
    return results.map((obj) => plainToInstance(YGOProCard, obj));
  }
  private queryInAllDBs(value: string, matchCount?: number) {
    return _.sortBy(
      _.uniqBy(
        _.flatten(this.dbs.map((db) => this.queryInDB(value, db))),
        (c) => c.id,
      ),
      (c) => c.getDistanceFrom(value),
    ).slice(0, matchCount || this.config.matchCount);
  }
  private async loadDBs(SQL: SQL.SqlJsStatic) {
    const allDatabaseFiles = _.flatten(
      await Promise.all(
        this.config.databasePaths.map(async (p) =>
          (await fs.readdir(p))
            .filter((f) => f.endsWith('.cdb'))
            .map((f) => path.join(p, f)),
        ),
      ),
    );
    this.logger.info(
      `Will load cards from ${allDatabaseFiles.length} databases.`,
    );
    const buffers = await Promise.all(
      allDatabaseFiles.map((f) => fs.readFile(f)),
    );
    this.dbs = buffers.map((b) => new SQL.Database(b));
  }

  async onConnect() {
    const SQL = await initSqlJs();
    return this.loadDBs(SQL);
  }

  onDisconnect() {
    this.dbs.forEach((db) => db.close());
  }

  @UseCommand('{{commandName}} [query]')
  @CommandLocale('zh', localeZh.cardCommand)
  @CommandLocale('en', localeEn.cardCommand)
  async onCardCommand(
    @PutArg(0) query: string,
    @PutOption('count', '-c <count:posint>') count: number,
    @PutSession() session: Session,
  ) {
    if (!query) {
      await session.send(session.text('.prompt-input'));
      query = await session.prompt();
    }
    if (!query) return;
    if (!count) {
      count = this.config.matchCount;
    }
    const cards = this.queryInAllDBs(query, count);
    if (!cards.length) {
      return session.text('.not-found');
    }
    if (cards.length === 1) {
      return cards[0].getDisplayString(this.config);
    }
    const itemLines = cards.map((c, i) => `${i + 1}. ${c.getIdAndName()}`);
    const borderLength = Math.max(...itemLines.map((l) => l.length)) + 1;
    await session.send(
      session.text('.select-prefix-part1') +
        query +
        session.text('.select-prefix-part2') +
        '\n' +
        _.repeat('-', borderLength) +
        '\n' +
        itemLines.join('\n') +
        '\n' +
        _.repeat('-', borderLength) +
        '\n' +
        session.text('.select-suffix'),
    );
    const reply = await session.prompt();
    if (!reply) {
      return;
    }
    const replyNumber = parseInt(reply);
    const card =
      cards[replyNumber - 1] ||
      cards.find((c) => c.id === replyNumber) ||
      cards.find((c) => c.name === reply) ||
      cards.find((c) => c.name.includes(reply)) ||
      cards.find((c) => c.desc.includes(reply));
    if (!card) {
      return session.text('.not-found');
    }
    return card.getDisplayString(this.config);
  }
}
