// import 'source-map-support/register';
import { WyWallBacklistPluginConfig } from './config';
import {
  CommandExample,
  CommandLocale,
  CRenderer,
  DefinePlugin,
  If,
  Inject,
  InjectLogger,
  Isolate,
  OnGuild,
  OnPlatform,
  OnSelection,
  PluginDef,
  Provide,
  PutArg,
  PutCommonRenderer,
  StarterPlugin,
  UseCommand,
  UseEvent,
  UseMiddleware,
  UsePlugin,
} from 'koishi-thirdeye';
import { Logger, Next, Quester, Session } from 'koishi';
import { BlacklistReturn } from './types';
export * from './config';

@OnPlatform('onebot')
@DefinePlugin()
class ManageHandler extends StarterPlugin() {
  @Inject()
  private wybl: Partial<WyWallBacklistPlugin>;

  async handleAccount(session: Session, skipAdminCheck = false) {
    if (!skipAdminCheck) {
      const selfInfo = await session.bot.getGuildMember(
        session.guildId,
        session.selfId,
      );
      if (
        !selfInfo ||
        (!selfInfo.roles.includes('admin') && !selfInfo.roles.includes('owner'))
      ) {
        return [];
      }
    }
    return await this.wybl.checkAccount(session.userId);
  }

  @UseMiddleware(true)
  async onGuildMessage(session: Session, next: Next) {
    if (session.subtype !== 'group') return next();
    const result = await this.handleAccount(session);
    if (result.length) {
      await session.bot.deleteMessage(session.channelId, session.messageId);
      await session.bot.kickGuildMember(session.guildId, session.userId);
      return [
        `用户 ${session.username}(${session.userId}) 位于蔷蔷黑名单，已被踢出群聊。`,
        '',
        ...result,
      ].join('\n');
    }
    return next();
  }

  @UseEvent('guild-member-request')
  async onGuildMemberRequest(session: Session) {
    const result = await this.handleAccount(session, true);
    if (result.length) {
      await session.bot.handleGuildMemberRequest(
        session.messageId,
        false,
        result.join(', '),
      );
      await session.bot.sendMessage(session.guildId, [
        `用户 ${session.userId} 位于蔷蔷黑名单，已禁止入群。`,
        '',
        ...result,
      ]);
    }
  }
}

@Isolate('wybl')
@Provide('wybl', { immediate: true })
@DefinePlugin({ name: 'wyblacklist' })
export default class WyWallBacklistPlugin extends StarterPlugin(
  WyWallBacklistPluginConfig,
) {
  @Inject(true)
  private http: Quester;

  @InjectLogger()
  private logger: Logger;

  @If<WyWallBacklistPlugin>((p) => p.config.autoManage)
  @UsePlugin()
  loadManage() {
    this.logger.info(`Auto manage is enabled.`);
    return PluginDef(ManageHandler);
  }

  async checkAccount(account: string | number) {
    const result = await this.http.get<BlacklistReturn>(
      `${this.config.endpoint}/api/blacklist`,
      {
        responseType: 'json',
        params: {
          recordsPerPage: 100,
          account,
        },
      },
    );
    return result.data.map(
      (item) =>
        `${new Date(item.blacklist.time).toLocaleString()} ${
          item.blacklist.content
        }`,
    );
  }

  @UseCommand('wybl <account:posint>')
  @CommandExample('wybl 123456789')
  @CommandLocale('zh', {
    description: '蔷蔷挂人查询',
    messages: {
      fail: '查询失败。',
      safe: '用户 {account} 是安全的。',
      unsafe: '用户 {account} 存在风险。',
      invalid: '请输入正确的 QQ 号。',
    },
  })
  @CommandLocale('en', {
    description: 'Query the blacklist of Wenyuan Wall.',
    messages: {
      fail: 'Failed to query blacklist.',
      safe: 'User {account} is safe.',
      unsafe: 'User {account} is risky.',
      invalid: 'Please input a valid QQ number.',
    },
  })
  async wybl(@PutArg(0) account: number, @PutCommonRenderer() r: CRenderer) {
    if (account < 10000 || account > 9999999999) {
      return r('.invalid', { account });
    }

    try {
      const result = await this.checkAccount(account);
      if (!result.length) {
        return r('.safe', { account });
      }
      return [r('.unsafe', { account }), '', ...result].join('\n');
    } catch (e) {
      this.logger.error(`Failed to fetch blacklist of ${account}: ${e}`);
      return r('.fail');
    }
  }
}
