import { WhoViewsPluginConfig } from './config';
import {
  Apply,
  DefinePlugin,
  Get,
  Inject,
  InjectLogger,
  KoaContext,
  PutSession,
  StarterPlugin,
  UseCommand,
} from 'koishi-thirdeye';
import LimQqwry from 'lim-qqwry';
import PicsContainer from 'koishi-plugin-pics';
import { Logger, Quester, Random, segment, Session } from 'koishi';
export * from './config';
import { OneBotBot } from '@koishijs/plugin-adapter-onebot';

interface WhoSession extends Quester.File {
  ips: string[];
  data: Buffer;
}

@DefinePlugin({ name: 'whoviews' })
export default class WhoViewsPlugin extends StarterPlugin(
  WhoViewsPluginConfig,
) {
  private db: LimQqwry;

  @InjectLogger()
  private logger: Logger;

  @Inject(true)
  private pics: PicsContainer;

  @Inject(true)
  private http: Quester;

  @Apply()
  private loadDb() {
    this.db = new LimQqwry(this.config.qqwryPath);
  }

  private sessions = new Map<string, WhoSession>();

  private async getPicUrl() {
    const pic = await this.pics.randomPic(
      this.config.tags,
      this.config.sourceTags,
    );
    if (!pic) {
      this.logger.warn(
        `no pic found for ${this.config.tags?.join(',') || '*'} from ${
          this.config.sourceTags?.join(',') || '*'
        }`,
      );
      return;
    }
    return pic.url;
  }

  @Get('{{path}}/:id')
  private async getPic(ctx: KoaContext) {
    const whoSession = this.sessions.get(ctx.params.id);
    if (!whoSession) {
      ctx.status = 404;
      return (ctx.body = 'Not found');
    }
    whoSession.ips.push(ctx.ip);
    ctx.type = whoSession.mime;
    ctx.body = whoSession.data;
    ctx.set('content-length', whoSession.data.length.toString());
    ctx.set('cache-control', 'no-cache');
    ctx.status = 200;
  }

  @UseCommand('whoviews', '谁在窥屏？')
  private async whoViews(@PutSession() session: Session) {
    const picUrl = await this.getPicUrl();
    if (!picUrl) return '服务似乎开小差了。';
    try {
      const info = await this.http.file(picUrl);
      const id = `${Random.id(20)}.${info.filename.split('.').pop()}`;
      this.sessions.set(id, { ...info, ips: [], data: Buffer.from(info.data) });
      const pathWithoutSlash = this.config.path.replace(/^\//, '');
      const sessionUrl = `${
        this.config.selfUrl || this.ctx.root.config.selfUrl
      }/${pathWithoutSlash}/${id}`;
      const messageSuffix = `让我 ${this.config.waitTime} 秒后康康谁在窥屏。`;
      if (session.platform === 'onebot') {
        const bot = session.bot as OneBotBot;
        const messageOnebot = `[CQ:xml,data=<?xml version="1.0" encoding="UTF-8" standalone="yes"?><msg serviceID="1"><item><title>${messageSuffix}</title></item><source name="窥屏检测" icon="${sessionUrl}" action="" appid="-1" /></msg>]`;
        if (session.guildId) {
          await bot.internal.sendGroupMsg(session.guildId, messageOnebot);
        } else {
          await bot.internal.sendPrivateMsg(session.userId, messageOnebot);
        }
      } else {
        await session.send(
          `${segment.image(sessionUrl).toString()}${messageSuffix}`,
        );
      }
      await new Promise((resolve) =>
        setTimeout(resolve, this.config.waitTime * 1000),
      );
      const ips = this.sessions.get(id)?.ips || [];
      this.sessions.delete(id);
      if (!ips.length) return '没人窥屏。';
      return `有 ${ips.length} 人窥屏：\n\n${ips
        .map((ip, i) => {
          const data = this.db.searchIP(ip);
          return `${i + 1}. ${data['ip']} ${data['country']} ${data['isp']}`;
        })
        .join('\n')}`;
    } catch (e) {
      this.logger.error(`Failed to process whoviews: ${e.message}`);
      return '服务似乎开小差了。';
    }
  }
}
