import { Context, Fork } from 'koishi';
import TypeORMPlugin from '../src';
import {
  Column,
  DataSource,
  Entity,
  EntityManager,
  ManyToOne,
  OneToMany,
  PrimaryGeneratedColumn,
  Repository,
} from 'typeorm';
import * as os from 'os';

@Entity()
class User {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;

  @OneToMany((type) => Book, (book) => book.author)
  books: Book[];
}

@Entity()
class Book {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;

  @ManyToOne((type) => User, (user) => user.books)
  author: User;
}

@Entity()
class Magic {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;
}

describe('Koishi typeorm', () => {
  let app: Context;
  let pluginInstance: Fork;

  beforeEach(async () => {
    app = new Context();
    await app.start();
    pluginInstance = app.plugin(TypeORMPlugin, {
      type: 'sqlite',
      database: `${os.tmpdir()}/${Math.random()}.sqlite`,
    });
  });

  afterEach(async () => {
    pluginInstance.dispose();
    await app.stop();
  });

  it('should create connection', async () => {
    expect(app.typeorm).toBeTruthy();
    await app.typeorm.create('foo', [User, Book]);
    const connection = app.typeorm.getDataSource('foo');
    expect(connection).toBeInstanceOf(DataSource);
    const manager = app.typeorm.getEntityManager('foo');
    expect(manager).toBeInstanceOf(EntityManager);
    const userRepo = app.typeorm.getRepository(User);
    expect(userRepo).toBeInstanceOf(Repository);
    let user = new User();
    user.name = 'Shigma';
    user = await userRepo.save(user);
    expect(typeof user.id).toBe('number');
    const gotUser = await userRepo.findOne({ where: { name: 'Shigma' } });
    expect(gotUser.name).toBe('Shigma');
  });

  it('should able to open more than 1 connection', async () => {
    await app.typeorm.create('fooo', [User, Book]);
    await app.typeorm.create('barr', [Magic]);
    const connection1 = app.typeorm.getDataSource('fooo');
    const connection2 = app.typeorm.getDataSource('barr');
    expect(connection1).toBeInstanceOf(DataSource);
    expect(connection2).toBeInstanceOf(DataSource);
    const userRepo = app.typeorm.getRepository(User);
    const magicRepo = app.typeorm.getRepository(Magic);
    expect(userRepo).toBeInstanceOf(Repository);
    expect(magicRepo).toBeInstanceOf(Repository);
    expect(userRepo.manager).toEqual(connection1.manager);
    expect(magicRepo.manager).toEqual(connection2.manager);
  });

  it('should connect with custom options', async () => {
    await app.typeorm.create('baz', [User, Book], {
      entityPrefix: 'fooooo_',
      metadataTableName: 'fooooo_metadata',
    });
    const connection = app.typeorm.getDataSource('baz');
    expect(connection.metadataTableName).toBe('fooooo_metadata');
  });

  it('should auto dispose when plugin gone', async () => {
    let instance: Fork;
    await new Promise<void>((resolve) => {
      instance = app.plugin(async (ctx) => {
        await ctx.typeorm.create('bar', [User, Book]);
        resolve();
      });
    });
    expect(app.typeorm.getDataSource('bar')).toBeInstanceOf(DataSource);
    expect(app.typeorm.getRepository(User)).toBeInstanceOf(Repository);
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    instance.dispose();
    await new Promise<void>((resolve) => {
      setTimeout(resolve, 1);
    });
    expect(app.typeorm.getDataSource('bar')).toBeUndefined();
    expect(app.typeorm.getRepository(User)).toBeUndefined();
  });

  it('should reload the same connection', async () => {
    await app.typeorm.create('yuzu', [User, Book]);
    expect(app.typeorm.getDataSource('yuzu')).toBeInstanceOf(DataSource);
    await app.typeorm.close('yuzu');
    expect(app.typeorm.getDataSource('yuzu')).toBeUndefined();
    await app.typeorm.create('yuzu', [User, Book]);
    expect(app.typeorm.getDataSource('yuzu')).toBeInstanceOf(DataSource);
  });
});
