// import 'source-map-support/register';
import {
  TypeORMEntity,
  TypeORMPluginConfig,
} from './config';
import {
  DefinePlugin,
  StarterPlugin,
  Caller,
  Provide,
  LifecycleEvents,
  PluginSchema,
} from 'koishi-thirdeye';
import { Connection, ConnectionOptions } from 'typeorm';
import { Context } from 'koishi';
export * from './config';
export * from 'typeorm';

interface ConnectionEntry {
  connection: Connection;
  entities: TypeORMEntity[];
}

declare module 'koishi' {
  interface Context {
    typeorm: TypeORMPlugin;
  }
}

@PluginSchema(TypeORMPluginConfig)
@Provide('typeorm', { immediate: true })
@DefinePlugin()
export default class TypeORMPlugin
  extends StarterPlugin(TypeORMPluginConfig)
  implements LifecycleEvents
{
  private registryMap = new Map<string, ConnectionEntry>();
  private entityMap = new Map<TypeORMEntity, string>();

  @Caller()
  private caller: Context;

  async close(token: string) {
    const entry = this.registryMap.get(token)!;
    if (!entry) return;
    const { connection, entities } = entry;
    await connection.close();
    this.registryMap.delete(token);
    for (const entity of entities) {
      this.entityMap.delete(entity);
    }
  }

  getConnection(token: string) {
    return this.registryMap.get(token)?.connection;
  }

  getEntityManager(token: string) {
    return this.getConnection(token)?.manager;
  }

  getRepository<T>(entity: TypeORMEntity<T>) {
    const token = this.entityMap.get(entity);
    if (!token) return;
    const connection = this.getConnection(token);
    if (!connection) return;
    return connection.getRepository(entity);
  }

  async create(
    token: string,
    entities: TypeORMEntity[],
    extraOptions: Partial<ConnectionOptions> = {},
  ) {
    if (this.registryMap.has(token)) {
      return this.getConnection(token);
    }
    const ctx = this.caller;
    ctx.on('dispose', () => this.close(token));

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const connectionOptions: ConnectionOptions = {
      ...this.config,
      entities,
      entityPrefix: `koishi_${token}_`,
      metadataTableName: `koishi_typeorm_metadata_${token}`,
      name: token,
      ...extraOptions,
    };
    const connection = await new Connection(connectionOptions).connect();
    this.registryMap.set(token, { connection, entities });
    for (const entity of entities) {
      this.entityMap.set(entity, token);
    }
    return connection;
  }

  async onDisconnect() {
    const tokens = Array.from(this.registryMap.keys());
    await Promise.all(tokens.map((token) => this.close(token)));
    this.registryMap.clear();
    this.entityMap.clear();
  }
}
