import { App } from 'koishi';
import TypeORMPlugin from '../src';
import {
  Column,
  Connection,
  Entity,
  EntityManager,
  ManyToOne,
  OneToMany,
  PrimaryGeneratedColumn,
  Repository,
} from 'typeorm';

@Entity()
class User {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;

  @OneToMany((type) => Book, (book) => book.author)
  books: Book[];
}

@Entity()
class Book {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;

  @ManyToOne((type) => User, (user) => user.books)
  author: User;
}

@Entity()
class Magic {
  @PrimaryGeneratedColumn()
  id: number;
  @Column('varchar', { length: 32 })
  name: string;
}

describe('Koishi typeorm', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    await app.start();
    app.plugin(TypeORMPlugin, {
      type: 'postgres',
      host: 'localhost',
      port: 5432,
      username: 'koishi',
      password: 'koishi@114514',
      database: 'koishi',
      //dropSchema: true,
    });
  });

  it('should create connection', async () => {
    expect(app.typeorm).toBeTruthy();
    await app.typeorm.create('foo', [User, Book]);
    const connection = app.typeorm.getConnection('foo');
    expect(connection).toBeInstanceOf(Connection);
    const manager = app.typeorm.getEntityManager('foo');
    expect(manager).toBeInstanceOf(EntityManager);
    const userRepo = app.typeorm.getRepository(User);
    expect(userRepo).toBeInstanceOf(Repository);
    let user = new User();
    user.name = 'Shigma';
    user = await userRepo.save(user);
    expect(typeof user.id).toBe('number');
    const gotUser = await userRepo.findOne({ where: { name: 'Shigma' } });
    expect(gotUser.name).toBe('Shigma');
  });

  it('should able to open more than 1 connection', async () => {
    await app.typeorm.create('fooo', [User, Book]);
    await app.typeorm.create('barr', [Magic]);
    const connection1 = app.typeorm.getConnection('fooo');
    const connection2 = app.typeorm.getConnection('barr');
    expect(connection1).toBeInstanceOf(Connection);
    expect(connection2).toBeInstanceOf(Connection);
    const userRepo = app.typeorm.getRepository(User);
    const magicRepo = app.typeorm.getRepository(Magic);
    expect(userRepo).toBeInstanceOf(Repository);
    expect(magicRepo).toBeInstanceOf(Repository);
    expect(userRepo.manager).toEqual(connection1.manager);
    expect(magicRepo.manager).toEqual(connection2.manager);
  });

  it('should connect with custom options', async () => {
    await app.typeorm.create('baz', [User, Book], {
      entityPrefix: 'fooooo_',
      metadataTableName: 'fooooo_metadata',
    });
    const connection = app.typeorm.getConnection('baz');
    expect(connection.metadataTableName).toBe('fooooo_metadata');
  });

  it('should auto dispose when plugin gone', async () => {
    await new Promise<void>((resolve) => {
      app.plugin(async (ctx) => {
        await ctx.typeorm.create('bar', [User, Book]);
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        ctx.on('fooo', () => ctx.dispose());
        resolve();
      });
    });
    expect(app.typeorm.getConnection('bar')).toBeInstanceOf(Connection);
    expect(app.typeorm.getRepository(User)).toBeInstanceOf(Repository);
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    app.emit('fooo');
    await new Promise<void>((resolve) => {
      setTimeout(resolve, 1);
    });
    expect(app.typeorm.getConnection('bar')).toBeUndefined();
    expect(app.typeorm.getRepository(User)).toBeUndefined();
  });

  afterEach(async () => {
    await app.stop();
  });
});
