import { TypeORMEntity, TypeORMPluginConfig } from './config';
import {
  DefinePlugin,
  StarterPlugin,
  Caller,
  Provide,
  LifecycleEvents,
  PluginSchema,
} from 'koishi-thirdeye';
import { DataSource, DataSourceOptions } from 'typeorm';
import { Context } from 'koishi';
export * from './config';
export * from 'typeorm';

interface DataSourceEntry {
  dataSource: DataSource;
  entities: TypeORMEntity[];
}

declare module 'koishi' {
  interface Context {
    typeorm: TypeORMPlugin;
  }
}

@Provide('typeorm', { immediate: true })
@DefinePlugin()
export default class TypeORMPlugin
  extends StarterPlugin(TypeORMPluginConfig)
  implements LifecycleEvents
{
  private registryMap = new Map<string, DataSourceEntry>();
  private entityMap = new Map<TypeORMEntity, string>();

  @Caller()
  private caller: Context;

  async close(token: string) {
    const entry = this.registryMap.get(token)!;
    if (!entry) return;
    const { dataSource, entities } = entry;
    // await dataSource.destroy();
    this.registryMap.delete(token);
    for (const entity of entities) {
      this.entityMap.delete(entity);
    }
  }

  getDataSource(token: string) {
    return this.registryMap.get(token)?.dataSource;
  }

  getEntityManager(token: string) {
    return this.getDataSource(token)?.manager;
  }

  getRepository<T>(entity: TypeORMEntity<T>) {
    const token = this.entityMap.get(entity);
    if (!token) return;
    const dataSource = this.getDataSource(token);
    if (!dataSource) return;
    return dataSource.getRepository(entity);
  }

  async create(
    token: string,
    entities: TypeORMEntity[],
    extraOptions: Partial<DataSourceOptions> = {},
  ) {
    if (this.registryMap.has(token)) {
      return this.getDataSource(token);
    }
    const ctx = this.caller;
    ctx.on('dispose', () => this.close(token));

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const dataSourceOptions: DataSourceOptions = {
      ...this.config,
      entities,
      entityPrefix: `koishi_${token}_`,
      metadataTableName: `koishi_typeorm_metadata_${token}`,
      name: token,
      ...extraOptions,
    };
    const dataSource = await new DataSource(dataSourceOptions).initialize();
    this.registryMap.set(token, { dataSource, entities });
    for (const entity of entities) {
      this.entityMap.set(entity, token);
    }
    return DataSource;
  }

  async onDisconnect() {
    const tokens = Array.from(this.registryMap.keys());
    await Promise.all(tokens.map((token) => this.close(token)));
    this.registryMap.clear();
    this.entityMap.clear();
  }
}
