import 'source-map-support/register';
import { Random, sleep, Cache } from 'koishi';
import { ThesaurusPluginConfig } from './config';
import { promises as fs } from 'fs';
import _ from 'lodash';
import {
  BasePlugin,
  DefinePlugin,
  Inject,
  LifecycleEvents,
} from 'koishi-thirdeye';

export interface ChatSession {
  name: string;
  playedWords: string[];
}

export * from './config';

declare module 'koishi' {
  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Cache {
    interface Tables {
      thesaurusChatSession: ChatSession;
    }
  }
}

@DefinePlugin({ name: 'thesaurus', schema: ThesaurusPluginConfig })
export default class ThesaurusPlugin
  extends BasePlugin<ThesaurusPluginConfig>
  implements LifecycleEvents
{
  @Inject(true)
  private cache: Cache;

  private wordData: Record<string, string[]> = {};
  private async loadWords(): Promise<void> {
    const path = this.config.path;
    try {
      if (path.match(/^https?:\/\//)) {
        this.wordData = await this.ctx.http.get(path);
      } else {
        this.wordData = JSON.parse(await fs.readFile(path, 'utf-8'));
      }
      this.ctx
        .logger('thesaurus')
        .info(
          `Loaded ${
            Object.keys(this.wordData).length
          } patterns from source ${path} .`,
        );
    } catch (e) {
      this.ctx
        .logger('thesaurus')
        .error(`Failed to load words from source ${path}: ${e.toString()}`);
      await sleep(5000);
      return this.loadWords();
    }
  }

  private async getFromCache(sessionId: string) {
    if (!sessionId) {
      return;
    }
    return this.cache.get('thesaurusChatSession', sessionId);
  }
  private async setToCache(sessionId: string, chatSession: ChatSession) {
    if (!sessionId) {
      return;
    }
    return this.cache.set(
      'thesaurusChatSession',
      sessionId,
      chatSession,
      this.config.chatTimeout,
    );
  }
  private async clearCacheOf(sessionId: string) {
    if (!sessionId) {
      return;
    }
    return this.cache.del('thesaurusChatSession', sessionId);
  }

  private async replyWith(
    word: string,
    sessionId: string,
    chatSession: ChatSession,
  ) {
    if (!word) {
      return;
    }
    chatSession.playedWords.push(word);
    if (chatSession.playedWords.length > this.config.trackingLength) {
      chatSession.playedWords.shift();
    }
    await this.setToCache(sessionId, chatSession);
    return word.replace(/你/g, chatSession.name).trim();
  }

  private tryPatterns(patterns: string[], usedWords: Set<string>) {
    if (!patterns.length) {
      return;
    }
    const pattern = Random.pick(patterns);
    const availableWords = this.wordData[pattern].filter(
      (w) => !usedWords.has(w),
    );
    if (!availableWords.length) {
      const shrinkedPatterns = patterns.filter((p) => p !== pattern);
      return this.tryPatterns(shrinkedPatterns, usedWords);
    }
    return Random.pick(availableWords);
  }

  private async replyChat(
    word: string,
    sessionId: string,
    chatSession: ChatSession,
  ) {
    const matchingPatterns = Object.keys(this.wordData).filter((w) =>
      word.includes(w),
    );
    if (!matchingPatterns.length) {
      const allWords = _.flatten(Object.values(this.wordData));
      let remainingAllWords = allWords.filter(
        (w) => !chatSession.playedWords.includes(w),
      );
      if (!remainingAllWords.length) {
        remainingAllWords = allWords;
      }
      return this.replyWith(
        Random.pick(remainingAllWords),
        sessionId,
        chatSession,
      );
    }
    const replyWord =
      this.tryPatterns(
        matchingPatterns,
        new Set<string>(chatSession.playedWords),
      ) || this.tryPatterns(matchingPatterns, new Set<string>());
    return this.replyWith(replyWord, sessionId, chatSession);
  }
  private formatName(name: string) {
    if (name.match(/^[a-zA-Z0-9]/)) {
      name = ` ${name}`;
    }
    if (name.match(/[a-zA-Z0-9]$/)) {
      name = `${name} `;
    }
    return name;
  }

  async onConnect() {
    const ctx = this.ctx;
    this.cache.table('thesaurusChatSession', {
      maxAge: this.config.chatTimeout,
    });
    await this.loadWords();
    ctx.middleware(async (session, next) => {
      const sessionId = `${session.platform}.${session.selfId}.${session.userId}`;
      const chatSession = await this.getFromCache(sessionId);
      if (chatSession) {
        //ctx
        //  .logger('thesaurus')
        //  .warn(`${session.userId} in chat ${JSON.stringify(chatSession)}`);
        if (session.content === 'quit') {
          await this.clearCacheOf(sessionId);
          await session.send('记得下次再找我聊喔~');
          return;
        }
        const reply = await this.replyChat(
          session.content,
          sessionId,
          chatSession,
        );
        await session.send(reply);
      }
      //ctx.logger('thesaurus').warn(`${session.userId} outside session`);
      return next();
    });
    ctx
      .command('startchat', '开始聊天')
      .usage('聊天开始后， quit 停止聊天。')
      .example('startchat --name Shigma 可以使用 Shigma 作为自己的名字来聊天。')
      .option('name', '--name <name:string>')
      .userFields(['name'])
      .action(async (argv) => {
        const { session, options } = argv;
        const sessionId = `${session.platform}.${session.selfId}.${session.userId}`;
        let name = session.username;
        if (options.name) {
          name = options.name;
        }
        if (!name) {
          name = '你';
        }
        const formattedName = this.formatName(name);
        await this.setToCache(sessionId, {
          name: formattedName,
          playedWords: [],
        });
        return `${formattedName}，开始和我聊天吧。聊累了的话，输入 quit 就可以结束聊天哦。`.trim();
      });
  }
}
