import 'source-map-support/register';
import { Context, Schema, Random } from 'koishi';
import { MyPluginConfig, MyPluginConfigLike } from './config';
import { schemaFromClass, schemaTransform } from 'koishi-utils-schemagen';
import { promises as fs } from 'fs';
import { sleep } from 'koishi';
import LRUCache from 'lru-cache';
import _ from 'lodash';

interface ChatSession {
  name: string;
  playedWords: string[];
}

export class MyPlugin {
  private config: MyPluginConfig;
  private ctx: Context;
  private wordData: Record<string, string[]> = {};
  private useDatabase = false;
  private async loadWords(): Promise<void> {
    const path = this.config.path;
    try {
      if (path.match(/^https?:\/\//)) {
        this.wordData = await this.ctx.http.get(path);
      } else {
        this.wordData = JSON.parse(await fs.readFile(path, 'utf-8'));
      }
      this.ctx
        .logger('thesaurus')
        .info(
          `Loaded ${
            Object.keys(this.wordData).length
          } patterns from source ${path} .`,
        );
    } catch (e) {
      this.ctx
        .logger('thesaurus')
        .error(`Failed to load words from source ${path}: ${e.toString()}`);
      await sleep(5000);
      return this.loadWords();
    }
  }
  // should be replaced with Koishi cache in future version
  private lru: LRUCache<string, ChatSession>;
  private async getFromCache(userId: string) {
    if (!userId) {
      return;
    }
    return this.lru.get(userId);
  }
  private async setToCache(userId: string, chatSession: ChatSession) {
    if (!userId) {
      return;
    }
    return this.lru.set(userId, chatSession, this.config.chatTimeout);
  }
  private async clearCacheOf(userId: string) {
    if (!userId) {
      return;
    }
    return this.lru.del(userId);
  }

  private async replyWith(
    word: string,
    userId: string,
    chatSession: ChatSession,
  ) {
    if (!word) {
      return;
    }
    chatSession.playedWords.push(word);
    if (chatSession.playedWords.length > this.config.trackingLength) {
      chatSession.playedWords.shift();
    }
    await this.setToCache(userId, chatSession);
    return word.replace(/你/g, chatSession.name).trim();
  }

  private tryPatterns(patterns: string[], usedWords: Set<string>) {
    if (!patterns.length) {
      return;
    }
    const pattern = Random.pick(patterns);
    const availableWords = this.wordData[pattern].filter(
      (w) => !usedWords.has(w),
    );
    if (!availableWords.length) {
      const shrinkedPatterns = patterns.filter((p) => p !== pattern);
      return this.tryPatterns(shrinkedPatterns, usedWords);
    }
    return Random.pick(availableWords);
  }

  private async replyChat(
    word: string,
    userId: string,
    chatSession: ChatSession,
  ) {
    const matchingPatterns = Object.keys(this.wordData).filter((w) =>
      word.includes(w),
    );
    if (!matchingPatterns.length) {
      const allWords = _.flatten(Object.values(this.wordData));
      let remainingAllWords = allWords.filter(
        (w) => !chatSession.playedWords.includes(w),
      );
      if (!remainingAllWords.length) {
        remainingAllWords = allWords;
      }
      return this.replyWith(
        Random.pick(remainingAllWords),
        userId,
        chatSession,
      );
    }
    const replyWord =
      this.tryPatterns(
        matchingPatterns,
        new Set<string>(chatSession.playedWords),
      ) || this.tryPatterns(matchingPatterns, new Set<string>());
    return this.replyWith(replyWord, userId, chatSession);
  }
  private formatName(name: string) {
    if (name.match(/^[a-zA-Z0-9]/)) {
      name = ` ${name}`;
    }
    if (name.match(/[a-zA-Z0-9]$/)) {
      name = `${name} `;
    }
    return name;
  }
  name = 'thesaurus-main';
  schema: Schema<MyPluginConfigLike> = schemaFromClass(MyPluginConfig);
  async apply(ctx: Context, config: MyPluginConfigLike) {
    this.ctx = ctx;
    this.config = schemaTransform(MyPluginConfig, config);
    ctx.on('connect', () => {
      if (ctx.database && this.config.useDatabase) this.useDatabase = true;
    });
    this.lru = new LRUCache<string, ChatSession>({
      maxAge: this.config.chatTimeout,
    });
    await this.loadWords();
    ctx.middleware(async (session, next) => {
      const chatSession = await this.getFromCache(session.userId);
      if (chatSession) {
        //ctx
        //  .logger('thesaurus')
        //  .warn(`${session.userId} in chat ${JSON.stringify(chatSession)}`);
        if (session.content === 'quit') {
          await this.clearCacheOf(session.userId);
          return session.send('记得下次再找我聊喔~');
        }
        const reply = await this.replyChat(
          session.content,
          session.userId,
          chatSession,
        );
        return session.send(reply);
      }
      //ctx.logger('thesaurus').warn(`${session.userId} outside session`);
      return next();
    });
    ctx
      .command('startchat', '开始聊天')
      .usage('聊天开始后， quit 停止聊天。')
      .example('startchat --name Shigma 可以使用 Shigma 作为自己的名字来聊天。')
      .option('name', '--name <name:string>')
      .userFields(['name'])
      .action(async (argv) => {
        const { session, options } = argv;
        const userId = session.userId;
        let name =
          session.author?.nickname ||
          session.author?.username ||
          session.username;
        if (this.useDatabase) {
          if (session.user.name) {
            name = session.user.name;
          }
        }
        if (options.name) {
          name = options.name;
        }
        if (!name) {
          name = '你';
        }
        const formattedName = this.formatName(name);
        await this.setToCache(userId, {
          name: formattedName,
          playedWords: [],
        });
        return `${formattedName}，开始和我聊天吧。聊累了的话，输入 quit 就可以结束聊天哦。`.trim();
      });
  }
}
