import 'source-map-support/register';
import { Context, Cache, Session } from 'koishi';
import { TabulatePluginConfig, TabulatePluginConfigLike } from './config';
import {
  KoishiPlugin,
  InjectConfig,
  Inject,
  OnApply,
  UseEvent,
  UseCommand,
  CommandUsage,
  CommandExample,
  PutSession,
  PutOption,
  CommandOption,
  PutArg,
  OnConnect,
} from 'koishi-thirdeye';
import { Game, ReportScoreResult } from './def/Match';
import { classToPlain, plainToClass } from 'class-transformer';
import _ from 'lodash';
export * from './config';

declare module 'koishi' {
  namespace Cache {
    interface Tables {
      tabulateMatches: Record<string, Partial<Game>>;
    }
  }
}

@KoishiPlugin({ name: 'tabulate', schema: TabulatePluginConfig })
export default class TabulatePlugin implements OnApply {
  constructor(private ctx: Context, config: TabulatePluginConfigLike) {}

  async getGames(key: string) {
    const dataFromCache = (await this.cache.get('tabulateMatches', key)) || {};
    return _.mapValues(dataFromCache, (d) => plainToClass(Game, d));
  }

  async saveGame(game: Game) {
    const dataFromCache =
      (await this.cache.get('tabulateMatches', game.location)) || {};
    dataFromCache[game.getKey()] = classToPlain(game);
    return this.cache.set('tabulateMatches', game.location, dataFromCache);
  }

  async deleteGame(game: Game) {
    const dataFromCache =
      (await this.cache.get('tabulateMatches', game.location)) || {};
    if (!dataFromCache[game.getKey()]) {
      return;
    }
    delete dataFromCache[game.getKey()];
    return this.cache.set('tabulateMatches', game.location, dataFromCache);
  }

  @InjectConfig()
  private config: TabulatePluginConfig;

  @Inject('cache', true)
  private cache: Cache;

  onApply() {
    this.cache.table('tabulateMatches', { maxAge: this.config.saveTime });
  }

  @UseCommand('tabulate', '排表')
  @CommandUsage('创建一个新的比赛，并进行排表操作。')
  @CommandExample(
    'tabulate --kof --teama AA --teamb BB --playera "Nanahira,Momobako,Miko" --playerb "Yuzurisa,Senya,Maika" 进行一个 KOF 对局的排表',
  )
  async onTabulate(
    @PutSession() session: Session,
    @PutOption('location', '-p <location:string>  比赛地点，默认为群内。')
    location: string,
    @PutOption('rule', '-r <rule:string>  比赛模式', { fallback: '2/3【KOF】' })
    rule: string,
    @PutOption('kof', '--kof  使用 KOF 赛制')
    useKof: boolean,
    @PutOption('head', '--head  使用 人头 赛制')
    useHead: boolean,
    @PutOption('teama', '--teama <teama:string>  A 队队名', { fallback: 'A' })
    teama: string,
    @PutOption('teamb', '--teamb <teamb:string>  B 队队名', { fallback: 'B' })
    teamb: string,
    @PutOption(
      'playera',
      '--playera <playera:string>  A 队队员名称，逗号分隔',
      {
        fallback: 'a1,a2,a3',
      },
    )
    playera: string,
    @PutOption(
      'playerb',
      '--playerb <playera:string>  B 队队员名称，逗号分隔',
      {
        fallback: 'b1,b2,b3',
      },
    )
    playerb: string,
  ) {
    const exactLocation =
      location || session.guildId || session.channelId || session.userId;
    if (!exactLocation) {
      return '无法获取比赛地点，请使用 --location 参数手动指定。';
    }
    const parsedPlayerA = playera.split(',');
    const parsedPlayerB = playerb.split(',');
    if (
      !parsedPlayerA.length ||
      !parsedPlayerB.length ||
      parsedPlayerA.length !== parsedPlayerB.length
    ) {
      return '两队人数不一致，无法排表。';
    }
    const game = new Game(
      exactLocation,
      useKof ? '2/3【KOF】' : useHead ? '人头赛' : rule,
      teama,
      teamb,
      parsedPlayerA,
      parsedPlayerB,
    );
    await this.saveGame(game);
    return game.format();
  }

  @UseCommand('tabulate/score <self:integer> <oppo:integer>', '上报比分')
  @CommandUsage('进行比分的更新操作，并视情况进行下一轮或结束比赛。')
  @CommandExample('score -n Nanahira 2 0 标记当局中 Nanahira 以 2:0 战胜对方。')
  async updateScore(
    @PutSession() session: Session,
    @PutOption('location', '-p <location:string>  比赛地点，默认为群内。')
    location: string,
    @PutOption('name', '-n <name:string>  自己的 ID')
    name: string,
    @PutArg(0)
    self: number,
    @PutArg(1)
    oppo: number,
  ) {
    const exactLocation =
      location || session.guildId || session.channelId || session.userId;
    const possibleNames: string[] = _.uniq(
      _.compact([
        name,
        session.author?.nickname,
        session.author?.username,
        session.username,
        session.userId,
      ]),
    );
    const gamesMap = await this.getGames(exactLocation);
    const game = Object.values(gamesMap).find((game) =>
      game.includesPendingPlayer(possibleNames),
    );
    if (!game) {
      return '未找到比赛。请尝试使用 -n <name> 指定自己的名称。';
    }
    const matchingName = game.includesPendingPlayer(possibleNames);
    const result = game.reportScore(matchingName, self, oppo);
    if (result === ReportScoreResult.NotFound) {
      return `未找到对局。`;
    }
    if (result === ReportScoreResult.Finish) {
      await this.deleteGame(game);
    } else {
      await this.saveGame(game);
    }
    return game.format();
  }
}
