import 'source-map-support/register';
import { Context } from 'koishi';
import { SRVProRoomInfo } from './def';
import { plainToClass } from 'class-transformer';
import { RegisterSchema, DefineSchema } from 'schemastery-gen';

export interface ServerLike {
  name: string;
  tags?: string[];
  endpoint: string;
  username?: string;
  password?: string;
  isDefault?: boolean;
  displayPrivateRooms?: boolean;
  displayPlayerIp?: boolean;
}

@RegisterSchema({ description: 'SRVPro 服务器' })
export class Server implements ServerLike {
  constructor(config: Partial<Server>) {}
  @DefineSchema({ description: '服务器名称', required: true })
  name: string;

  @DefineSchema({ description: '服务器搜索词', default: [] })
  tags: string[];

  @DefineSchema({
    description: '服务器 API 连接后台，不带任何路径和尾随斜杠',
    required: true,
  })
  endpoint: string;

  @DefineSchema({ description: '服务器后台用户名' })
  username: string;

  @DefineSchema({ description: '服务器后台密码' })
  password: string;

  @DefineSchema({ description: '是否是默认服务器', default: false })
  isDefault: boolean;

  @DefineSchema({ description: '是否显示私密房间', default: false })
  displayPrivateRooms: boolean;

  @DefineSchema({ description: '是否显示玩家 IP', default: false })
  displayPlayerIp: boolean;

  async fetchRooms(ctx: Context) {
    return ctx.http.get<Partial<SRVProRoomInfo>>(
      `${this.endpoint}/api/getrooms`,
      {
        username: this.username,
        pass: this.password,
      },
    );
  }
  async getRoomString(ctx: Context) {
    const info = await this.fetchRooms(ctx);
    if (!info) {
      return `获取 ${this.name} 房间列表失败。`;
    }
    const infoObj = plainToClass(SRVProRoomInfo, info);
    return infoObj.stringify(this);
  }
}

export interface MyPluginConfigLike {
  listCommandName?: string;
  roomCommandName?: string;
  servers: ServerLike[];
}

@RegisterSchema({ required: true })
export class MyPluginConfig implements MyPluginConfigLike {
  constructor(config: Partial<MyPluginConfig>) {}
  @DefineSchema({ description: '服务器列表命令名', default: 'serverlist' })
  listCommandName: string;

  @DefineSchema({ description: '房间列表命令名', default: 'roomlist' })
  roomCommandName: string;

  @DefineSchema({
    type: Server,
    required: true,
    description: 'SRVPro 服务器列表',
  })
  servers: Server[];

  getServer(value: string) {
    return (
      this.servers.find((s) => s.name === value) ||
      this.servers.find((s) => s.tags.includes(value)) ||
      this.servers.find((s) => s.name.includes(value)) ||
      this.servers.find((s) => s.endpoint === value) ||
      this.servers.find((s) => s.isDefault)
    );
  }
}
