import 'source-map-support/register';
import { Context, Logger, Quester, Session } from 'koishi';
import { ShadowbanPluginConfig, ShadowbanPluginConfigLike } from './config';
import {
  DefinePlugin,
  InjectConfig,
  UseCommand,
  PutArg,
  CommandUsage,
  CommandExample,
  Inject,
  PutSession,
  InjectLogger,
  CommandDescription,
} from 'koishi-thirdeye';
export * from './config';

interface ShadowbanResult {
  profile: Profile;
  tests: Tests;
}

interface Tests {
  ghost: Ghost;
  more_replies: Morereplies;
  search: string | false;
  typeahead: boolean;
}

interface Morereplies {
  ban: boolean;
  in_reply_to: string;
  tweet: string;
}

interface Ghost {
  ban: boolean;
}

interface Profile {
  exists: boolean;
  protected: boolean;
  screen_name: string;
  has_tweets: boolean;
  suspended?: boolean;
}

@DefinePlugin({ name: 'shadowban', schema: ShadowbanPluginConfig })
export default class ShadowbanPlugin {
  constructor(private ctx: Context, config: ShadowbanPluginConfigLike) {}
  @InjectConfig()
  private config: ShadowbanPluginConfig;

  @Inject('http', true)
  private http: Quester;

  @InjectLogger()
  private logger: Logger;

  private addLine<T>(
    obj: T,
    goodChecker: (o: T) => boolean,
    goodMessage: string,
    badMessage: string,
    failMessage: string,
  ): string {
    if (obj == null) {
      return '\n[x] ' + failMessage;
    }
    const good = goodChecker(obj);
    if (good) {
      return `\n[+] ${goodMessage}`;
    } else {
      return `\n[-] ${badMessage}`;
    }
  }

  @UseCommand('shadowban <username:string>', '测试 Twitter 帐号状态')
  @CommandDescription({
    en: 'Test Twitter account status',
    cn: '测试 Twitter 帐号状态',
  })
  @CommandUsage('测试需要一定的时间，请耐心等待。')
  @CommandExample('shadowban Nana_Yumesaki 获取 @Nana_Yumesaki 的状态')
  async testShadowban(
    @PutArg(0) username: string,
    @PutSession() session: Session,
  ) {
    await session.send(`正在测试 @${username} 的状态...`);
    let result: ShadowbanResult;
    try {
      result = await this.http.get<ShadowbanResult>(
        this.config.getUrlForUsername(username),
      );
    } catch (e) {
      this.logger.error(
        `Failed to fetch profile for @${username}: ${e.toString()}`,
      );
      return `无法获取 @${username} 的状态，请稍后再试。`;
    }
    if (!result.profile?.exists) {
      return `@${username} 不存在。`;
    }
    let output = `@${username} 的状态:`;
    if (result.profile.protected) {
      output += '\n* 该帐号已被设置为受保护。';
    }
    if (!result.profile.has_tweets) {
      output += '\n* 该帐号没有发布过推文。';
    }
    if (result.profile.suspended) {
      output += '\n* 该帐号已被封禁。';
    }
    if (!result.tests) {
      return output;
    }
    output += '\n';
    const tests = result.tests;
    output += this.addLine(
      tests.typeahead,
      (t) => t,
      'No search suggestion ban. (未被封禁搜索建议)',
      'Search suggestion ban. (封禁搜索建议)',
      '无法获取搜索建议封禁状态',
    );
    output += this.addLine(
      tests.search,
      (t) => !!t,
      'No search ban. (未被封禁搜索)',
      'Search ban. (封禁搜索)',
      '无法获取搜索封禁状态',
    );
    output += this.addLine(
      tests.ghost?.ban,
      (t) => !t,
      'No ghost ban. (未被封禁)',
      'Ghost ban. (封禁)',
      '无法获取封禁状态',
    );
    output += this.addLine(
      tests.more_replies?.ban,
      (t) => !t,
      'No reply deboosting detected. (未被隐藏回复)',
      'Reply deboosting detected. (隐藏回复)',
      '无法获取隐藏回复状态',
    );
    return output;
  }
}
