// import 'source-map-support/register';
import { RetweetNotifierPluginConfig } from './config';
import {
  DefineModel,
  DefinePlugin,
  Inject,
  ModelField,
  Primary,
  Reusable,
  UseModel,
} from 'koishi-thirdeye';
import { SchedulePlugin } from 'koishi-schedule-send';
import { Database, Quester, Random, Tables } from 'koishi';
import moment from 'moment';
import _ from 'lodash';
export * from './config';

declare module 'koishi' {
  interface Tables {
    retweeted: Retweeted;
  }
}

export interface Tweet {
  cashtags: string[];
  conversation_id: string;
  datestamp: string;
  datetime: string;
  geo: string;
  hashtags: string[];
  id: number;
  id_str: string;
  lang: string;
  likes_count: number;
  link: string;
  mentions: string[];
  name: string;
  near: string;
  photos: string[];
  place: string;
  quote_url: string;
  replies_count: number;
  reply_to: string[];
  retweet: boolean;
  retweet_date: string;
  retweet_id: string;
  retweets_count: number;
  source: string;
  thumbnail: string;
  timestamp: string;
  timezone: string;
  trans_dest: string;
  trans_src: string;
  translate: string;
  tweet: string;
  urls: string[];
  user_id: number;
  user_id_str: string;
  user_rt: string;
  user_rt_id: string;
  username: string;
  video: number;
}

const asyncFilter = async <T>(
  arr: T[],
  predicate: (v: T) => boolean | Promise<boolean>,
) => {
  const results = await Promise.all(arr.map(predicate));

  return arr.filter((_v, index) => results[index]);
};

@DefineModel('retweeted')
export class Retweeted {
  @Primary()
  @ModelField({
    type: 'string',
    length: 20,
  })
  id: string;

  @ModelField({
    type: 'json',
  })
  payload: Tweet;
}

@Reusable()
@UseModel(Retweeted)
@DefinePlugin({ name: 'retweet-notifier' })
export default class RetweetNotifierPlugin extends SchedulePlugin(
  RetweetNotifierPluginConfig,
) {
  @Inject(true)
  http: Quester;

  @Inject(true)
  database: Database<Tables>;

  private async notNotified(id: string) {
    const existing = await this.database.get('retweeted', { id }, ['id']);
    return !existing?.length;
  }

  async send() {
    let { data: tweets } = await this.http.post<{ data: Tweet[] }>(
      this.config.endpoint + '/Search',
      {
        Username: this.config.username,
        Since: moment()
          .subtract(this.config.traceDays, 'days')
          .utc()
          .format('YYYY-MM-DD HH:mm:ss'),
        Search: this.config.search || undefined,
      },
      {
        headers: {
          Authorization: 'Bearer ' + this.config.token,
        },
      },
    );
    // tweets = tweets.filter((t) => t.retweets_count === 0); // notify non-retweets only
    tweets = await asyncFilter(tweets, (tweet) =>
      this.notNotified(tweet.id_str),
    );
    if (!tweets.length) return;
    const leastRetweet = _.minBy(
      tweets,
      (t) => (t.retweets_count << 16) | (t.replies_count << 8) | t.likes_count, // pick the least retweeted tweet
    );

    await this.database.upsert('retweeted', [
      {
        id: leastRetweet.id_str,
        payload: leastRetweet,
      },
    ]);

    return leastRetweet.link + '\n' + Random.pick(this.config.message);
  }
}
