// import 'source-map-support/register';
import { RailgunPluginConfig } from './config';
import {
  DefinePlugin,
  StarterPlugin,
  Inject,
  PutArg,
  PutOption,
  UseCommand,
  CommandUsage,
  PutValue,
  PutObject,
  For,
  CommandTemplate,
  PutRenderer,
  Renderer,
} from 'koishi-thirdeye';
import { omit, Quester } from 'koishi';
export * from './config';

interface Railgun {
  gateways: Gateway[];
  user: User;
  router: string;
  destinations: Destination[];
}

interface Destination {
  name: string;
  desc: string;
}

interface User {
  ip: string;
  destination: string;
  remoteGatewayId: number;
  localGatewayId?: any;
}

interface Gateway {
  id: number;
  name: string;
  description: string;
  type: string;
  destType: string;
}

interface RailgunResult {
  success: boolean;
  statusCode: number;
  message: string;
}

type AxiosRequestConfig = Parameters<Quester['get']>[1];

class IPInput {
  @PutOption(
    'ip',
    '--ip <ip>  指定 IP 地址。需要在 Railgun Enterprise 具有相关权限。',
  )
  ip: string;

  getHeader() {
    return this.ip ? { 'x-forwarded-for': this.ip } : {};
  }
}

class Input extends IPInput {
  @PutArg(0, { required: false })
  value: string;

  getData(field: string) {
    return { [field]: this.value === 'clear' ? null : this.value };
  }
}

const RailgunUsage = CommandUsage(
  '查询可用列表，或者设置设置一个值。输入 clear 清除值。',
);

@DefinePlugin()
export default class RailgunPlugin extends StarterPlugin(RailgunPluginConfig) {
  @Inject(true)
  private http: Quester;

  private async railgunData(extras: AxiosRequestConfig = {}) {
    return await this.http.get<Railgun>(
      `${this.config.endpoint}/api/data`,
      extras,
    );
  }

  private async handleCommand(
    input: Input,
    field: string,
    query: (r: Railgun) => string,
  ) {
    const data = await this.railgunData({ headers: input.getHeader() });
    if (!input.value) {
      return query(data);
    }
    const postData = {
      ...omit(data.user, ['ip']),
      ...input.getData(field),
    };
    try {
      const result = await this.http.post<RailgunResult>(
        `${this.config.endpoint}/api/select`,
        postData,
        {
          headers: input.getHeader(),
          validateStatus: () => true,
        },
      );
      if (result?.success) {
        return '操作成功。';
      } else {
        return `操作失败: ${result?.message}`;
      }
    } catch (e) {
      `操作失败: ${e.message}`;
    }
  }

  @UseCommand('railgun', 'Railgun Enterprise', { empty: true })
  private railgun() {}

  @UseCommand('railgun.current', '查询当前状态')
  @CommandTemplate(
    'current',
    '{user.ip}@{router}\n出境网络: {destination}\n出境网关: {remoteGateway}\n本地网关: {localGateway}',
  )
  private async status(
    @PutObject() ipInput: IPInput,
    @PutRenderer('.current')
    template: Renderer<
      Railgun & {
        destination: string;
        remoteGateway: string;
        localGateway: string;
      }
    >,
  ) {
    const data = await this.railgunData({ headers: ipInput.getHeader() });
    return template({
      ...data,
      remoteGateway:
        data.gateways.find((g) => g.id === data.user.remoteGatewayId)
          ?.description || '无',
      localGateway:
        data.gateways.find((g) => g.id === data.user.localGatewayId)
          ?.description || '自动',
      destination:
        data.destinations.find((d) => d.name === data.user.destination)?.desc ||
        '直连',
    });
  }

  @For(() => [
    { type: 'local', desc: '本地' },
    { type: 'remote', desc: '出境' },
  ])
  @UseCommand('railgun.{{type}}', '{{desc}}网关')
  @RailgunUsage
  private gatewayCommand(
    @PutObject() input: Input,
    @PutValue('{{type}}') type: string,
  ) {
    const field = `${type}GatewayId`;
    return this.handleCommand(input, field, (r) =>
      r.gateways
        .filter((g) => g.destType === type)
        .map(
          (g) =>
            `${g.id === r.user[field] ? '>' : ' '} ${g.id} - ${g.description}`,
        )
        .join('\n'),
    );
  }

  @UseCommand('railgun.destination', '出境网络')
  @RailgunUsage
  railgunDestination(@PutObject() input: Input) {
    return this.handleCommand(input, 'destination', (r) =>
      r.destinations
        .map(
          (d) =>
            `${d.name === r.user.destination ? '>' : ' '} ${d.name} - ${
              d.desc
            }`,
        )
        .join('\n'),
    );
  }
}
