import 'source-map-support/register';
import { Context, Quester, Cache, Logger, Random } from 'koishi';
import { PicsourceHeisiConfig, PicsourceHeisiConfigLike } from './config';
import {
  KoishiPlugin,
  InjectConfig,
  Inject,
  OnApply,
  OnConnect,
  InjectLogger,
} from 'koishi-thirdeye';
import { PicsContainer, PicSource } from 'koishi-plugin-pics';
export * from './config';

declare module 'koishi' {
  interface Modules {
    'picsource-heisi': typeof import('.');
  }

  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Cache {
    interface Tables {
      heisiList: string[];
    }
  }
}

@KoishiPlugin({ name: 'picsource-heisi', schema: PicsourceHeisiConfig })
export default class PicsourceHeisi
  extends PicSource
  implements OnApply, OnConnect {
  constructor(ctx: Context, config: PicsourceHeisiConfigLike) {
    super(ctx);
  }
  @InjectConfig()
  private config: PicsourceHeisiConfig;

  @Inject('http')
  private http: Quester;

  @Inject('pics', true)
  private pics: PicsContainer;

  @Inject('cache', true)
  private cache: Cache;

  @InjectLogger()
  private logger: Logger;

  onApply() {
    this.config.applyTo(this);
    this.cache.table('heisiList', { maxAge: this.config.ttl });
  }

  async onConnect() {
    this.logger.info(`Initializing pic source ${this.config.name}...`);
    await this.getRandomUrl();
    this.pics.addSource(this, this.ctx);
  }

  async randomPic() {
    const url = await this.getRandomUrl();
    if (!url) return;
    return {
      url,
      description: this.config.picDescription,
    };
  }

  async getRandomUrl() {
    const list = await this.cache.get('heisiList', 'list');
    if (!list || !list.length) {
      await this.fetchPicList();
    }
    if (!list || !list.length) {
      return;
    }
    return Random.pick(list);
  }

  async fetchPicList() {
    this.logger.info(`Fetching pic list from ${this.config.endpoint}...`);
    try {
      const content = await this.http.get(this.config.endpoint);
      const urls = content
        .trim()
        .split('\n')
        .map((line) => line.trim())
        .filter((line) => line.startsWith('http'));
      await this.cache.set('heisiList', 'list', urls);
      this.logger.info(
        `fetched ${urls.length} pics from ${this.config.endpoint}`,
      );
    } catch (e) {
      this.logger.error(
        `Failed to fetch pic list from ${
          this.config.endpoint
        }: ${e.toString()}`,
      );
    }
  }
}
