// import 'source-map-support/register';
import { Session, Next, segment } from 'koishi';
import { OrderPickerConfig } from './config';
import {
  DefinePlugin,
  Inject,
  UseMiddleware,
  OnGuild,
  OnPlatform,
  UseCommand,
  StarterPlugin,
  Isolate,
  UsingService,
  PutSelfId,
  UsePlugin,
  PluginDef,
  OnPrivate,
  Provide,
} from 'koishi-thirdeye';
export * from './config';
import AragamiPlugin, { CacheKey, CacheTTL } from 'koishi-plugin-cache-aragami';

export class SelfOnline {
  @CacheKey()
  selfId: string;

  online: boolean;
}

@CacheTTL(60000)
export class LastMessage {
  @CacheKey()
  guildId: string;
  userId: string;
  content: string;

  fromSession(session: Session) {
    this.guildId = session.guildId;
    this.userId = session.userId;
    this.content = session.content;
    return this;
  }
}

const base = StarterPlugin(OrderPickerConfig);

@Provide('utility', { immediate: true })
@DefinePlugin()
class Utility extends base {
  @Inject()
  private aragami: AragamiPlugin;

  async isOnline(selfId: string) {
    const online = await this.aragami.get(SelfOnline, selfId);
    return !!online?.online;
  }

  async setOnline(selfId: string, online: boolean) {
    await this.aragami.set(SelfOnline, { selfId, online });
  }

  async saveLastMessage(session: Session) {
    const message = new LastMessage().fromSession(session);
    const previousSaving = await this.aragami.get(LastMessage, session.guildId);
    if (previousSaving && session.userId === previousSaving.userId) {
      message.content = previousSaving.content + '\n' + message.content;
    }
    return this.aragami.set(message);
  }

  async fetchLastMessage(session: Session) {
    const lastMessage = await this.aragami.get(LastMessage, session.guildId);
    if (lastMessage) {
      await this.aragami.del(LastMessage, session.guildId);
      if (lastMessage.userId === session.userId) {
        return lastMessage.content;
      }
    }
  }
}

@DefinePlugin()
class ControlPanel extends base {
  @Inject(true)
  private utility: Utility;

  @UseCommand('order', '抢单管理', { empty: true })
  orderCommand() {}

  private async switch(selfId: string, online: boolean, desc: string) {
    const currentStatus = await this.utility.isOnline(selfId);
    if (currentStatus === online) {
      return `我已经在${desc}了。`;
    }
    await this.utility.setOnline(selfId, online);
    return `${desc}成功。`;
  }

  @UseCommand('order.on', '开启接单')
  async switchOn(@PutSelfId() id: string) {
    return this.switch(id, true, '上班');
  }

  @UseCommand('order.off', '关闭接单')
  async switchOff(@PutSelfId() id: string) {
    return this.switch(id, false, '下班');
  }

  @UseCommand('order.status', '查看上班状态')
  async status(@PutSelfId() id: string) {
    const online = await this.utility.isOnline(id);
    return `我${online ? '上班' : '下班'}了。`;
  }
}

@UsingService('aragami')
@Isolate('utility')
@OnPlatform('onebot')
@DefinePlugin()
export default class OrderPicker extends base {
  @Inject()
  private utility: Utility;

  @UsePlugin()
  loadUtility() {
    return PluginDef(Utility, this.config);
  }

  @OnPrivate('{{masterId}}')
  @UsePlugin()
  loadControlPanel() {
    return PluginDef(ControlPanel, this.config);
  }

  @OnGuild()
  @UseMiddleware()
  async detectOrder(session: Session, next: Next) {
    if (!this.config.isMatchingTarget(session)) return next();
    if (!this.config.acceptMessage(session)) {
      await this.utility.saveLastMessage(session);
      return next();
    }
    const lastMessageContent = await this.utility.fetchLastMessage(session);
    if (!(await this.utility.isOnline(session.selfId))) return next();
    let description = segment
      .parse(session.content)
      .filter((s) => s.type === 'text')
      .map((s) => s.data.content)
      .join('')
      .trim();

    if (!description) {
      description = lastMessageContent || '';
    }

    await session.bot.sendPrivateMessage(
      this.config.masterId,
      `接到群 ${session.guildId} 的由 ${session.userId} 发布的单: ${description}`,
    );
    await this.utility.setOnline(session.selfId, false);
    return this.config.pickWord;
  }
}
