import 'source-map-support/register';
import { Context, Cache, Session, Next, Bot, segment } from 'koishi';
import { OrderPickerConfig, OrderPickerConfigLike } from './config';
import {
  DefinePlugin,
  InjectConfig,
  Inject,
  UseMiddleware,
  OnGuild,
  OnPlatform,
  OnApply,
} from 'koishi-thirdeye';
import moment from 'moment';
export * from './config';

export interface PickedOrderInfo {
  sourceGuildId: string;
  sourceUserId: string;
  sourceUserName: string;
  description: string;
  time: number;
}

declare module 'koishi' {
  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Cache {
    interface Tables {
      orderPickerOnline: boolean;
      pickedOrderInfo: PickedOrderInfo;
    }
  }
}

@OnPlatform('onebot')
@DefinePlugin({ name: 'order-picker', schema: OrderPickerConfig })
export default class OrderPicker implements OnApply {
  constructor(private ctx: Context, config: OrderPickerConfigLike) {}
  @InjectConfig()
  private config: OrderPickerConfig;

  @Inject(true)
  private cache: Cache;

  onApply() {
    this.cache.table('orderPickerOnline', {});
    this.cache.table('pickedOrderInfo', {
      maxAge: 1000 * 60 * this.config.cooldown,
    });
    const cmd = this.ctx
      .private(this.config.masterId)
      .command('order-picker', '抢单管理');
    cmd
      .subcommand('.status', '抢单状态')
      .usage('获取抢单状态')
      .action(async ({ session }) => {
        const online = await this.isOnline(session.selfId);
        const orderInfo = await this.getLastInfo(session.selfId);
        const firstLine = `抢单状态：${online ? '上班' : '下班'}`;
        let secondLine = '';
        if (orderInfo) {
          const time = moment.unix(orderInfo.time);
          const toTime = time.clone().add(this.config.cooldown, 'minutes');
          secondLine = `来自群：${orderInfo.sourceGuildId}\n发布人：${
            orderInfo.sourceUserName
          }(${orderInfo.sourceUserId})\n描述：${
            orderInfo.description
          }\n时间：${time.format(
            'YYYY-MM-DD HH:mm:ss',
          )}\n结束时间：${toTime.format('YYYY-MM-DD HH:mm:ss')}`;
        } else {
          secondLine = '最近没有抢单';
        }
        return `${firstLine}\n${secondLine}`;
      });
    cmd
      .subcommand('.on', '上班')
      .usage('开启抢单')
      .action(async ({ session }) => {
        const online = await this.isOnline(session.selfId);
        if (online) {
          return '已经在上班了。';
        }
        await this.setOnline(session.selfId, true);
        return '上班成功。';
      });
    cmd
      .subcommand('.off', '下单')
      .usage('关闭抢单')
      .action(async ({ session }) => {
        const online = await this.isOnline(session.selfId);
        if (!online) {
          return '还没有上班。';
        }
        await this.setOnline(session.selfId, false);
        return '下班成功。';
      });
    cmd
      .subcommand('.clear', '清除当前单')
      .usage('强制清除当前单')
      .action(async ({ session }) => {
        const orderInfo = await this.getLastInfo(session.selfId);
        if (!orderInfo) {
          return '当前没有单。';
        }
        await this.removeLastInfo(session.selfId);
        return '清除成功。';
      });
  }

  private async getLastInfo(selfId: string) {
    return this.cache.get('pickedOrderInfo', selfId);
  }

  private async setLastInfo(selfId: string, info: PickedOrderInfo) {
    return this.cache.set('pickedOrderInfo', selfId, info);
  }

  private async removeLastInfo(selfId: string) {
    return this.cache.del('pickedOrderInfo', selfId);
  }

  private async isOnline(selfId: string) {
    return this.cache.get('orderPickerOnline', selfId);
  }

  private async setOnline(selfId: string, online: boolean) {
    return this.cache.set('orderPickerOnline', selfId, online);
  }

  @OnGuild()
  @UseMiddleware()
  async detectOrder(session: Session, next: Next) {
    if (!this.config.acceptMessage(session)) return next();
    if (!(await this.isOnline(session.selfId))) return next();
    if (await this.getLastInfo(session.selfId)) return next();
    const description = segment
      .parse(session.content)
      .filter((s) => s.type === 'text')
      .map((s) => s.data.content)
      .join('');
    await this.setLastInfo(session.selfId, {
      description,
      time: moment().unix(),
      sourceGuildId: session.guildId,
      sourceUserId: session.userId,
      sourceUserName:
        session.author?.nickname ||
        session.author?.username ||
        session.username ||
        session.userId,
    });
    await session.send(this.config.pickWord);
    await session.bot.sendPrivateMessage(
      this.config.masterId,
      `接到单:\n${description}`,
    );
    return;
  }
}
