// import 'source-map-support/register';
import {
  DefinePlugin,
  StarterPlugin,
  UseCommand,
  CommandUsage,
  CommandExample,
  PutOption,
  InjectLogger,
  CommandDescription,
  PutTemplate,
  Renderer,
} from 'koishi-thirdeye';
import { ping, NewPingResult } from 'minecraft-protocol';
import { Logger, segment } from 'koishi';

@DefinePlugin({ name: 'mcinfo' })
export default class MinecraftInfoPlugin extends StarterPlugin() {
  private getServerPrefix(result: NewPingResult) {
    let text = '';
    if (result.favicon?.startsWith('data:image/png;base64,')) {
      const base64Text = result.favicon.slice(22).replace(/\n/g, '');
      text += segment.image(`base64://${base64Text}`).toString();
      text += '\n';
    }
    return text;
  }
  private getServerSuffix(result: NewPingResult) {
    let text = '';
    if (result.description) {
      text += '\n\n';
      if (typeof result.description === 'string') {
        text += result.description;
      } else {
        text += `${result.description.text}${(result.description.extra || [])
          .map((m) => m?.text || '')
          .join('')}`;
      }
    }
    return text;
  }

  @InjectLogger()
  private logger: Logger;

  @UseCommand('mcinfo', '查询 Minecraft 服务器信息。')
  @CommandDescription({ en: 'Query Minecraft server info.' })
  @CommandUsage('查询指定的 Minecraft 服务器信息。')
  @CommandExample('mcinfo -s 2b2t.org')
  @CommandExample('mcinfo -s 2b2t.org -p 25565')
  async onQueryServer(
    @PutOption('host', '-s <host:string>  服务器地址', {
      description: { en: 'Server address' },
    })
    host: string,
    @PutOption('port', '-p <port:posint>  服务器端口，默认 25565', {
      description: { en: 'Server port' },
    })
    port: number,
    @PutTemplate('plzSpecifyHost', {
      zh: '请指定服务器地址。',
      en: 'Please specify server host.',
    })
    plzSpecifyHost: Renderer<void>,
    @PutTemplate('connectFail', {
      zh: '连接服务器 {hostString} 失败，请检查服务器地址是否正确。',
      en: 'Failed to connect to server {hostString}, please check the server address.',
    })
    connectFail: Renderer<{ hostString: string }>,
    @PutTemplate('serverInfo', {
      zh: `版本: {version.name}\n玩家: {players.online}/{players.max}\n{samplePlayers}\n延迟: {latency} ms\n`,
      en: `Version: {version.name}\nPlayers: {players.online}/{players.max}\n{samplePlayers}\nLatency: {latency} ms\n`,
    })
    serverInfo: Renderer<NewPingResult & { samplePlayers: string }>,
  ) {
    if (!host) {
      return plzSpecifyHost();
    }
    if (!port && host.includes(':')) {
      const splited = host.split(':');
      host = splited[0];
      port = parseInt(splited[1], 10) || 25565;
    }
    const hostString = port ? `${host}:${port}` : host;
    try {
      const result = (await ping({ host, port })) as NewPingResult;

      return `${hostString}\n${this.getServerPrefix(result)}${serverInfo({
        ...result,
        samplePlayers: (result.players?.sample || [])
          .map((player) => `* ${player.name}\n`)
          .join('\n'),
      })}${this.getServerSuffix(result)}`;
    } catch (e) {
      this.logger.error(
        `Failed to query Minecraft server ${hostString}: ${e.message}`,
      );
      return connectFail({ hostString });
    }
  }
}
