import 'source-map-support/register';
import {
  DefinePlugin,
  BasePlugin,
  UseCommand,
  CommandUsage,
  CommandExample,
  PutOption,
  InjectLogger,
} from 'koishi-thirdeye';
import { ping, NewPingResult } from 'minecraft-protocol';
import { Logger, segment } from 'koishi';

@DefinePlugin({ name: 'mcinfo' })
export default class MinecraftInfoPlugin extends BasePlugin<any> {
  private getServerText(result: NewPingResult) {
    let text = '';
    if (result.favicon?.startsWith('data:image/png;base64,')) {
      const base64Text = result.favicon.slice(22).replace(/\n/g, '');
      text += segment.image(`base64://${base64Text}`);
      text += '\n';
    }
    if (result.version) {
      text += `版本: ${result.version.name}\n`;
    }
    if (result.players) {
      text += `玩家: ${result.players.online}/${result.players.max}\n`;
      for (const player of result.players.sample || []) {
        text += `* ${player.name}\n`;
      }
    }
    text += `延迟: ${result.latency} ms`;
    if (result.description) {
      text += '\n\n';
      if (typeof result.description === 'string') {
        text += result.description;
      } else {
        text += `${result.description.text}${(result.description.extra || [])
          .map((m) => m?.text || '')
          .join('')}`;
      }
    }
    return text;
  }

  @InjectLogger()
  private logger: Logger;

  @UseCommand('mcinfo', '查询 Minecraft 服务器信息。')
  @CommandUsage('查询指定的 Minecraft 服务器信息。')
  @CommandExample('mcinfo -s 2b2t.org')
  @CommandExample('mcinfo -s 2b2t.org -p 25565')
  async onQueryServer(
    @PutOption('host', '-s <host:string>  服务器地址') host: string,
    @PutOption('port', '-p <port:posint>  服务器端口，默认 25565') port: number,
  ) {
    if (!host) {
      return '请指定服务器地址。';
    }
    if (!port && host.includes(':')) {
      const splited = host.split(':');
      host = splited[0];
      port = parseInt(splited[1], 10) || 25565;
    }
    const hostString = port ? `${host}:${port}` : host;
    try {
      const result = (await ping({ host, port })) as NewPingResult;
      return `${hostString}\n${this.getServerText(result)}`;
    } catch (e) {
      this.logger.error(
        `Failed to query Minecraft server ${hostString}: ${e.message}`,
      );
      return `连接服务器 ${hostString} 失败，请检查服务器地址是否正确。`;
    }
  }
}
