// import 'source-map-support/register';
import { LibVirtPluginConfig, LibVirtPluginConfigLike } from './config';
import {
  DefinePlugin,
  BasePlugin,
  InjectLogger,
  UseCommand,
  PutOption,
  CommandExample,
  LifecycleEvents,
  CommandDescription,
  Inject,
  PutTemplate,
  Renderer,
} from 'koishi-thirdeye';
import { Logger, Template } from 'koishi';
import _ from 'lodash';
import { DomainStateToText, getDomainsFromHypervisor } from './utils';
export * from './config';
import util from 'util';
import { execFile } from 'child_process';

const connectFailMessage = `未找到主机或连接失败，请检查配置或主机连接。`;

@DefinePlugin({ name: 'libvirt', schema: LibVirtPluginConfig })
export default class LibVirtPlugin
  extends BasePlugin<LibVirtPluginConfig, LibVirtPluginConfigLike>
  implements LifecycleEvents
{
  @InjectLogger()
  private logger: Logger;

  private async connect(name: string) {
    const hypervisor = this.config.findHypervisorDef(name);
    if (!hypervisor) return;
    try {
      return await hypervisor.toHypervisor();
    } catch (e) {
      this.logger.error(`Failed to connect to hypervisor ${name}`, e.message);
    }
  }

  @UseCommand('virt', '虚拟机管理', { empty: true })
  @CommandDescription({ en: 'Virtual machine management' })
  async virtCommand() {}

  @UseCommand('virt.hosts', '查看主机列表')
  @CommandDescription({ en: 'List hypervisors' })
  async virtHostsCommand(
    @PutOption('host', '-s <name:string>  指定主机', {
      description: { en: 'Specify host' },
    })
    host: string,
    @PutTemplate('noAvailableHosts', {
      zh: '没有可用的主机。',
      en: 'No available hosts.',
    })
    noAvailableHosts: Renderer<void>,
  ) {
    const defs = _.compact(
      host ? [this.config.findHypervisorDef(host)] : this.config.servers,
    );
    if (!defs.length) {
      return noAvailableHosts();
    }
    return (await Promise.all(defs.map((def) => def.hypervisorInfo()))).join(
      '\n',
    );
  }

  @UseCommand('virt.vm', '查看虚拟机列表')
  @CommandDescription({ en: 'List virtual machines' })
  @CommandExample('virt.vm -s localhost  查看 localhost 主机的虚拟机列表。')
  @CommandExample(
    'virt.vm -s localhost -m ubuntu  查看 localhost 主机的 ubuntu 虚拟机信息。',
  )
  async virtVmsCommand(
    @PutOption('host', '-s <name:string>  指定主机', {
      description: { en: 'Specify host' },
    })
    host: string,
    @PutOption('vm', '-m <name:string>  指定虚拟机', {
      description: { en: 'Specify VM' },
    })
    vm: string,
    @PutOption('displayUuid', '-u  显示 uuid', {
      description: { en: 'Display UUID' },
    })
    displayUuid: boolean,
    @PutTemplate('noAvailableVMs', {
      zh: '没有可用的虚拟机。',
      en: 'No available VMs.',
    })
    noAvailableVMs: Renderer<void>,
    @PutTemplate('list', {
      zh: '{name} 主机的虚拟机列表:',
      en: 'VMs of {name}:',
    })
    vmOf: Renderer<{ name: string }>,
  ) {
    const { name, hypervisor } = (await this.connect(host)) || {};
    if (!hypervisor) return connectFailMessage;
    let domains = await getDomainsFromHypervisor(hypervisor);
    if (vm) {
      domains = domains.filter(
        (d) => d.name === vm || d.id?.toString() === vm || d.uuid === vm,
      );
    }
    if (!domains.length) {
      await hypervisor.connectClose();
      return noAvailableVMs();
    }
    const result = domains
      .map(
        (d) =>
          `${d.id || '-'} ${d.name} ${DomainStateToText(d.info.state)} (${
            d.info.nrVirtCpu
          } vCPUs, ${d.info.memory / 1024} / ${d.info.maxMem / 1024} MiB)${
            displayUuid ? `\n  UUID: ${d.uuid}` : ''
          }`,
      )
      .join('\n');
    await hypervisor.connectClose();
    return `${vmOf({ name })}\n${result}`;
  }

  private async virtActionCommand(
    actionName: string,
    verb: string = actionName,
    host: string,
    vm: string,
  ) {
    if (!vm) return '请指定虚拟机。';
    const { name, uri, hypervisor } = (await this.connect(host)) || {};
    if (!hypervisor) return connectFailMessage;
    const domain = (await getDomainsFromHypervisor(hypervisor)).find(
      (d) => d.name === vm || d.id?.toString() === vm || d.uuid === vm,
    );
    await hypervisor.connectClose();
    if (!domain) {
      return `未找到虚拟机 ${vm} 。`;
    }
    try {
      const { stdout, stderr } = await util.promisify(execFile)(
        'virsh',
        ['--connect', uri, actionName, domain.name],
        { encoding: 'utf8' },
      );
      return `${name} 主机的虚拟机 ${vm} ${verb}成功:\n${stdout}${
        stderr ? `\n${stderr}` : ''
      }`;
    } catch (e) {
      const errorMessage = e.stderr || e.message;
      this.logger.error(
        `Failed to ${actionName} domain ${domain.name}: ${errorMessage}`,
      );
      return `${name} 主机的虚拟机 ${vm} ${verb}发生错误：${errorMessage}`;
    }
  }

  @Inject()
  private i18n: Template;

  private defineActionCommand(action: string, verb: string = action) {
    this.i18n.define('en', `commands.virt.${action}`, {
      description: `Perform ${action} on virtual machine`,
    });
    this.ctx
      .command(`virt.${action}`, `${verb}虚拟机`)
      .example(
        `virt.${action} -s localhost -m ubuntu  ${verb} localhost 上的 ubuntu 虚拟机。`,
      )
      .option('host', '-s <name:string>  指定主机')
      .option('vm', '-m <name:string>  指定虚拟机')
      .action((a) =>
        this.virtActionCommand(action, verb, a.options.host, a.options.vm),
      );
    return this;
  }

  onApply() {
    this.defineActionCommand('start', '启动')
      .defineActionCommand('shutdown', '关闭')
      .defineActionCommand('destroy', '强制关闭')
      .defineActionCommand('reset', '强制重启')
      .defineActionCommand('reboot', '重启')
      .defineActionCommand('suspend', '挂起')
      .defineActionCommand('resume', '恢复');
  }
}
