// import 'source-map-support/register';
import { DefineSchema, RegisterSchema } from 'koishi-thirdeye';
import libvirt from '@avaflow/libvirt';

@RegisterSchema()
export class HypervisorConfig {
  @DefineSchema({ required: true, description: '主机连接地址。' })
  uri: string;

  @DefineSchema({ required: true, description: '主机名称。' })
  name: string;

  @DefineSchema({ default: false, description: '是否为默认主机。' })
  isDefault?: boolean;

  async toHypervisor() {
    const hypervisor = new libvirt.Hypervisor({ uri: this.uri });
    await hypervisor.connectOpen();
    return { name: this.name, uri: this.uri, hypervisor };
  }

  async hypervisorInfo() {
    try {
      const { hypervisor } = await this.toHypervisor();
      const info = await hypervisor.nodeGetInfo();
      await hypervisor.connectClose();
      return `${this.name} (${info.model} ${info.cpus} CPUs ${
        info.memory / 1024
      } MB)`;
    } catch (e) {
      console.error(e);
      return `${this.name} (连接失败)`;
    }
  }
}

@RegisterSchema()
export class LibVirtPluginConfig {
  constructor(config: LibVirtPluginConfigLike) {}

  @DefineSchema({
    type: HypervisorConfig,
    default: [{ name: 'localhost', isDefault: true, uri: 'qemu:///system' }],
  })
  servers: HypervisorConfig[];

  @DefineSchema({ description: 'virsh 命令程序的路径。', default: 'virsh' })
  virshPath: string;

  findHypervisorDef(name: string) {
    if (!name) return this.servers.find((server) => server.isDefault);
    const exact = this.servers.find((server) => server.name === name);
    if (exact) return exact;
    const prefix = this.servers.find((server) => server.name.startsWith(name));
    if (prefix) return prefix;
    return this.servers.find((server) => server.name.includes(name));
  }
}

export interface LibVirtPluginConfigLike {
  servers?: { uri: string; name: string; isDefault?: boolean }[];
  virsh?: string;
}
